#!/usr/bin/env PYTHONHASHSEED=1234 python3

# Copyright 2014-2019 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import random
random.seed(1234)

import logging
from pprint import pprint
from sys import stdout as STDOUT

# Wygenerowanie wszystkich danych wyjściowych w katalogu tymczasowym.
import atexit
import gc
import io
import os
import tempfile

TEST_DIR = tempfile.TemporaryDirectory()
atexit.register(TEST_DIR.cleanup)

# Prawidłowe zakończenie procesów w systemie Windows.
OLD_CWD = os.getcwd()
atexit.register(lambda: os.chdir(OLD_CWD))
os.chdir(TEST_DIR.name)

def close_open_files():
    everything = gc.get_objects()
    for obj in everything:
        if isinstance(obj, io.IOBase):
            obj.close()

atexit.register(close_open_files)


# Przykład 1.
def normalize(numbers):
    total = sum(numbers)
    result = []
    for value in numbers:
        percent = 100 * value / total
        result.append(percent)
    return result


# Przykład 2.
visits = [15, 35, 80]
percentages = normalize(visits)
print(percentages)
assert sum(percentages) == 100.0


# Przykład 3.
path = 'my_numbers.txt'
with open(path, 'w') as f:
    for i in (15, 35, 80):
        f.write('%d\n' % i)

def read_visits(data_path):
    with open(data_path) as f:
        for line in f:
            yield int(line)


# Przykład 4.
it = read_visits('my_numbers.txt')
percentages = normalize(it)
print(percentages)


# Przykład 5.
it = read_visits('my_numbers.txt')
print(list(it))
print(list(it))  # Brak możliwości ponownego użycia.


# Przykład 6.
def normalize_copy(numbers):
    numbers_copy = list(numbers)  # Kopia iteratora.
    total = sum(numbers_copy)
    result = []
    for value in numbers_copy:
        percent = 100 * value / total
        result.append(percent)
    return result


# Przykład 7.
it = read_visits('my_numbers.txt')
percentages = normalize_copy(it)
print(percentages)
assert sum(percentages) == 100.0


# Przykład 8.
def normalize_func(get_iter):
    total = sum(get_iter())   # Nowy iterator.
    result = []
    for value in get_iter():  # Nowy iterator.
        percent = 100 * value / total
        result.append(percent)
    return result


# Przykład 9.
path = 'my_numbers.txt'
percentages = normalize_func(lambda: read_visits(path))
print(percentages)
assert sum(percentages) == 100.0


# Przykład 10.
class ReadVisits:
    def __init__(self, data_path):
        self.data_path = data_path

    def __iter__(self):
        with open(self.data_path) as f:
            for line in f:
                yield int(line)


# Przykład 11.
visits = ReadVisits(path)
percentages = normalize(visits)
print(percentages)
assert sum(percentages) == 100.0


# Przykład 12.
def normalize_defensive(numbers):
    if iter(numbers) is numbers:  # To jest iterator -- niedobrze!
        raise TypeError('Konieczne jest dostarczenie kontenera.')
    total = sum(numbers)
    result = []
    for value in numbers:
        percent = 100 * value / total
        result.append(percent)
    return result

visits = [15, 35, 80]
normalize_defensive(visits)  # Bez błędu.

it = iter(visits)
try:
    normalize_defensive(it)
except TypeError:
    pass
else:
    assert False


# Przykład 13.
from collections.abc import Iterator 

def normalize_defensive(numbers):
    if isinstance(numbers, Iterator):  # Inny sposób na sprawdzenie.
        raise TypeError('Konieczne jest dostarczenie kontenera.')
    total = sum(numbers)
    result = []
    for value in numbers:
        percent = 100 * value / total
        result.append(percent)
    return result

visits = [15, 35, 80]
normalize_defensive(visits)  # Bez błędu.

it = iter(visits)
try:
    normalize_defensive(it)
except TypeError:
    pass
else:
    assert False


# Przykład 14.
visits = [15, 35, 80]
percentages = normalize_defensive(visits)
assert sum(percentages) == 100.0

visits = ReadVisits(path)
percentages = normalize_defensive(visits)
assert sum(percentages) == 100.0


# Przykład 15.
try:
    visits = [15, 35, 80]
    it = iter(visits)
    normalize_defensive(it)
except:
    logging.exception('Wystąpił błąd w kodzie wywołującym.')
else:
    assert False
