#!/usr/bin/env python3

# Copyright 2014 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import logging
from pprint import pprint
from sys import stdout as STDOUT


# Przykład 1.
def download(item):
    return item

def resize(item):
    return item

def upload(item):
    return item


# Przykład 2.
from threading import Lock
from collections import deque

class MyQueue(object):
    def __init__(self):
        self.items = deque()
        self.lock = Lock()


# Przykład 3.
    def put(self, item):
        with self.lock:
            self.items.append(item)


# Przykład 4.
    def get(self):
        with self.lock:
            return self.items.popleft()


# Przykład 5.
from threading import Thread
from time import sleep

class Worker(Thread):
    def __init__(self, func, in_queue, out_queue):
        super().__init__()
        self.func = func
        self.in_queue = in_queue
        self.out_queue = out_queue
        self.polled_count = 0
        self.work_done = 0


# Przykład 6.
    def run(self):
        while True:
            self.polled_count += 1
            try:
                item = self.in_queue.get()
            except IndexError:
                sleep(0.01)  # Brak zadania do wykonania.
            except AttributeError:
                # Sygnał wyjścia.
                return
            else:
                result = self.func(item)
                self.out_queue.put(result)
                self.work_done += 1


# Przykład 7.
download_queue = MyQueue()
resize_queue = MyQueue()
upload_queue = MyQueue()
done_queue = MyQueue()
threads = [
    Worker(download, download_queue, resize_queue),
    Worker(resize, resize_queue, upload_queue),
    Worker(upload, upload_queue, done_queue),
]


# Przykład 8.
for thread in threads:
    thread.start()
for _ in range(1000):
    download_queue.put(object())


# Przykład 9.
import time
while len(done_queue.items) < 1000:
    # Zrób coś użytecznego podczas oczekiwania.
    time.sleep(0.1)
# Zatrzymanie wszystkich wątków przez zgłoszenie wyjątków,
# w ich metodach.
for thread in threads:
    thread.in_queue = None


# Przykład 10.
processed = len(done_queue.items)
polled = sum(t.polled_count for t in threads)
print('Przetworzono', processed, 'elementów po wykonaniu',
      polled, 'sprawdzeń')


# Przykład 11.
from queue import Queue
queue = Queue()

def consumer():
    print('Konsument oczekuje')
    queue.get()                # Uruchomienie po metodzie put() przedstawionej poniżej.
    print('Konsument zakończył pracę')

thread = Thread(target=consumer)
thread.start()


# Przykład 12.
print('Producent umieszcza dane')
queue.put(object())            # Uruchomienie przed metodą get() przedstawioną powyżej.
thread.join()
print('Producent zakończył pracę')


# Przykład 13.
queue = Queue(1)               # Bufor o wielkości 1.

def consumer():
    time.sleep(0.1)            # Oczekiwanie.
    queue.get()                # Drugie wywołanie.
    print('Konsument pobiera dane 1')
    queue.get()                # Czwarte wywołanie.
    print('Konsument pobiera dane 2')

thread = Thread(target=consumer)
thread.start()


# Przykład 14.
queue.put(object())            # Pierwsze wywołanie.
print('Producent umieszcza dane 1')
queue.put(object())            # Trzecie wywołanie.
print('Producent umieszcza dane 2')
thread.join()
print('Producent zakończył pracę')


# Przykład 15.
in_queue = Queue()

def consumer():
    print('Konsument oczekuje')
    work = in_queue.get()      # Zakończone jako drugie.
    print('Konsument pracuje')
    # Wykonywanie pracy.
    print('Konsument zakończył pracę')
    in_queue.task_done()       # Zakończone jako trzecie.

Thread(target=consumer).start()


# Przykład 16.
in_queue.put(object())         # Zakończone jako pierwsze.
print('Producent oczekuje')
in_queue.join()                # Zakończone jako czwarte.
print('Producent zakończył pracę')


# Przykład 17.
class ClosableQueue(Queue):
    SENTINEL = object()

    def close(self):
        self.put(self.SENTINEL)


# Przykład 18.
    def __iter__(self):
        while True:
            item = self.get()
            try:
                if item is self.SENTINEL:
                    return  # Powoduje zakończenie działania wątku.
                yield item
            finally:
                self.task_done()


# Przykład 19.
class StoppableWorker(Thread):
    def __init__(self, func, in_queue, out_queue):
        super().__init__()
        self.func = func
        self.in_queue = in_queue
        self.out_queue = out_queue

    def run(self):
        for item in self.in_queue:
            result = self.func(item)
            self.out_queue.put(result)


# Przykład 20.
download_queue = ClosableQueue()
resize_queue = ClosableQueue()
upload_queue = ClosableQueue()
done_queue = ClosableQueue()
threads = [
    StoppableWorker(download, download_queue, resize_queue),
    StoppableWorker(resize, resize_queue, upload_queue),
    StoppableWorker(upload, upload_queue, done_queue),
]


# Przykład 21.
for thread in threads:
    thread.start()
for _ in range(1000):
    download_queue.put(object())
download_queue.close()


# Przykład 22.
download_queue.join()
resize_queue.close()
resize_queue.join()
upload_queue.close()
upload_queue.join()
print(done_queue.qsize(), 'elementów zostało przetworzonych')
