#!/usr/bin/env python3

# Copyright 2014 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import logging
from pprint import pprint
from sys import stdout as STDOUT


# Przykład 1.
def normalize(numbers):
    total = sum(numbers)
    result = []
    for value in numbers:
        percent = 100 * value / total
        result.append(percent)
    return result


# Przykład 2.
visits = [15, 35, 80]
percentages = normalize(visits)
print(percentages)


# Przykład 3.
path = '/tmp/my_numbers.txt'
with open(path, 'w') as f:
    for i in (15, 35, 80):
        f.write('%d\n' % i)

def read_visits(data_path):
    with open(data_path) as f:
        for line in f:
            yield int(line)


# Przykład 4.
it = read_visits('/tmp/my_numbers.txt')
percentages = normalize(it)
print(percentages)


# Przykład 5.
it = read_visits('/tmp/my_numbers.txt')
print(list(it))
print(list(it))  # Brak możliwości ponownego użycia.


# Przykład 6.
def normalize_copy(numbers):
    numbers = list(numbers)  # Kopia iteratora.
    total = sum(numbers)
    result = []
    for value in numbers:
        percent = 100 * value / total
        result.append(percent)
    return result


# Przykład 7.
it = read_visits('/tmp/my_numbers.txt')
percentages = normalize_copy(it)
print(percentages)


# Przykład 8.
def normalize_func(get_iter):
    total = sum(get_iter())   # Nowy iterator.
    result = []
    for value in get_iter():  # Nowy iterator.
        percent = 100 * value / total
        result.append(percent)
    return result


# Przykład 9.
percentages = normalize_func(lambda: read_visits(path))
print(percentages)


# Przykład 10.
class ReadVisits(object):
    def __init__(self, data_path):
        self.data_path = data_path

    def __iter__(self):
        with open(self.data_path) as f:
            for line in f:
                yield int(line)


# Przykład 11.
visits = ReadVisits(path)
percentages = normalize(visits)
print(percentages)


# Przykład 12.
def normalize_defensive(numbers):
    if iter(numbers) is iter(numbers):  # To jest iterator -- nie dobrze!
        raise TypeError('Konieczne jest dostarczenie kontenera')
    total = sum(numbers)
    result = []
    for value in numbers:
        percent = 100 * value / total
        result.append(percent)
    return result


# Przykład 13.
visits = [15, 35, 80]
normalize_defensive(visits)  # Bez błędu.
visits = ReadVisits(path)
normalize_defensive(visits)  # Bez błędu.


# Przykład 14.
try:
    it = iter(visits)
    normalize_defensive(it)
except:
    logging.exception('Oczekiwany')
else:
    assert False
