#!/usr/bin/env python3

# Copyright 2014 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import logging
from pprint import pprint
from sys import stdout as STDOUT


# Przykład 1.
def my_coroutine():
    while True:
        received = yield
        print('Otrzymano:', received)

it = my_coroutine()
next(it)             # Wywołanie współprogramu.
it.send('Pierwszy')
it.send('Drugi')


# Przykład 2.
def minimize():
    current = yield
    while True:
        value = yield current
        current = min(value, current)


# Przykład 3.
it = minimize()
next(it)            # Wywołanie generatora.
print(it.send(10))
print(it.send(4))
print(it.send(22))
print(it.send(-1))


# Przykład 4.
ALIVE = '*'
EMPTY = '-'


# Przykład 5.
from collections import namedtuple
Query = namedtuple('Query', ('y', 'x'))


# Przykład 6.
def count_neighbors(y, x):
    n_ = yield Query(y + 1, x + 0)  # Północ.
    ne = yield Query(y + 1, x + 1)  # Północny wschód.
    # Zdefiniowanie kolejnych kierunków e_, se, s_, sw, w_, nw ...
    e_ = yield Query(y + 0, x + 1)  # Wschód.
    se = yield Query(y - 1, x + 1)  # Południowy wschód.
    s_ = yield Query(y - 1, x + 0)  # Południe.
    sw = yield Query(y - 1, x - 1)  # Południowy zachód.
    w_ = yield Query(y + 0, x - 1)  # Zachód.
    nw = yield Query(y + 1, x - 1)  # Północny zachód.
    neighbor_states = [n_, ne, e_, se, s_, sw, w_, nw]
    count = 0
    for state in neighbor_states:
        if state == ALIVE:
            count += 1
    return count


# Przykład 7.
it = count_neighbors(10, 5)
q1 = next(it)                  # Pobranie pierwszego obiektu.
print('Pierwsze wyrażenie yield: ', q1)
q2 = it.send(ALIVE)            # Wysłanie informacji o stanie q1, pobranie q2.
print('Drugie wyrażenie yield:', q2)
q3 = it.send(ALIVE)            # Wysłanie informacji o stanie q2, pobranie q3.
print('...')
q4 = it.send(EMPTY)
q5 = it.send(EMPTY)
q6 = it.send(EMPTY)
q7 = it.send(EMPTY)
q8 = it.send(EMPTY)
try:
    count = it.send(EMPTY)     # Wysłanie informacji o stanie q8, pobranie ostatecznej wartości licznika.
except StopIteration as e:
    print('Liczba: ', e.value)  # Wartość pochodząca z polecenia return.


# Przykład 8.
Transition = namedtuple('Transition', ('y', 'x', 'state'))


# Przykład 9.
def game_logic(state, neighbors):
    pass

def step_cell(y, x):
    state = yield Query(y, x)
    neighbors = yield from count_neighbors(y, x)
    next_state = game_logic(state, neighbors)
    yield Transition(y, x, next_state)


# Przykład 10.
def game_logic(state, neighbors):
    if state == ALIVE:
        if neighbors < 2:
            return EMPTY     # Śmierć: zbyt mało.
        elif neighbors > 3:
            return EMPTY     # Śmierć: zbyt wiele.
    else:
        if neighbors == 3:
            return ALIVE     # Regeneracja.
    return state


# Przykład 11.
it = step_cell(10, 5)
q0 = next(it)           # Obiekt Query położenia początkowego.
print('Ja:      ', q0)
q1 = it.send(ALIVE)     # Wysłanie mojego stanu, ustawienie pola sąsiada.
print('Q1:      ', q1)
print('...')
q2 = it.send(ALIVE)
q3 = it.send(ALIVE)
q4 = it.send(ALIVE)
q5 = it.send(ALIVE)
q6 = it.send(EMPTY)
q7 = it.send(EMPTY)
q8 = it.send(EMPTY)
t1 = it.send(EMPTY)     # Wysłanie stanu q8, podjęcie decyzji w grze.
print('Wynik: ', t1)


# Przykład 12.
TICK = object()

def simulate(height, width):
    while True:
        for y in range(height):
            for x in range(width):
                yield from step_cell(y, x)
        yield TICK


# Przykład 13.
class Grid(object):
    def __init__(self, height, width):
        self.height = height
        self.width = width
        self.rows = []
        for _ in range(self.height):
            self.rows.append([EMPTY] * self.width)

    def __str__(self):
        output = ''
        for row in self.rows:
            for cell in row:
                output += cell
            output += '\n'
        return output


# Przykład 14.
    def query(self, y, x):
        return self.rows[y % self.height][x % self.width]

    def assign(self, y, x, state):
        self.rows[y % self.height][x % self.width] = state


# Przykład 15.
def live_a_generation(grid, sim):
    progeny = Grid(grid.height, grid.width)
    item = next(sim)
    while item is not TICK:
        if isinstance(item, Query):
            state = grid.query(item.y, item.x)
            item = sim.send(state)
        else:  # Konieczne jest przekształcenie.
            progeny.assign(item.y, item.x, item.state)
            item = next(sim)
    return progeny


# Przykład 16.
grid = Grid(5, 9)
grid.assign(0, 3, ALIVE)
grid.assign(1, 4, ALIVE)
grid.assign(2, 2, ALIVE)
grid.assign(2, 3, ALIVE)
grid.assign(2, 4, ALIVE)
print(grid)


# Przykład 17.
class ColumnPrinter(object):
    def __init__(self):
        self.columns = []

    def append(self, data):
        self.columns.append(data)

    def __str__(self):
        row_count = 1
        for data in self.columns:
            row_count = max(row_count, len(data.splitlines()) + 1)
        rows = [''] * row_count
        for j in range(row_count):
            for i, data in enumerate(self.columns):
                line = data.splitlines()[max(0, j - 1)]
                if j == 0:
                    padding = ' ' * (len(line) // 2)
                    rows[j] += padding + str(i) + padding
                else:
                    rows[j] += line
                if (i + 1) < len(self.columns):
                    rows[j] += ' | '
        return '\n'.join(rows)

columns = ColumnPrinter()
sim = simulate(grid.height, grid.width)
for i in range(5):
    columns.append(str(grid))
    grid = live_a_generation(grid, sim)

print(columns)


# Przykład 20.
# To są dane dla diagramu początkowego.
grid = Grid(5, 5)
grid.assign(1, 1, ALIVE)
grid.assign(2, 2, ALIVE)
grid.assign(2, 3, ALIVE)
grid.assign(3, 3, ALIVE)

columns = ColumnPrinter()
sim = simulate(grid.height, grid.width)
for i in range(5):
    columns.append(str(grid))
    grid = live_a_generation(grid, sim)

print(columns)
