#!/usr/bin/env python3

# Copyright 2014 Brett Slatkin, Pearson Education Inc.
#
# Udostępniono na licencji Apache w wersji 2.0 ("Licencja").
# Tego pliku można używać jedynie zgodnie z warunkami Licencji.
# Treść Licencji znajdziesz na stronie:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# O ile obowiązujące prawo nie stanowi inaczej lub czegoś innego nie
# uzgodniono w formie pisemnej, oprogramowanie objęte Licencją jest
# dostarczane w stanie, w jakim jest (wersja "AS IS"), BEZ JAKIEJKOLWIEK
# GWARANCJI, ani wyrażonej otwarcie, ani domyślnej. Dokładne zasady
# i warunki Licencji znajdziesz w jej treści.

# Przygotowania mające na celu odtworzenie środowiska użytego w książce.
import logging
from pprint import pprint
from sys import stdout as STDOUT


# Przykład 1.
from datetime import datetime, timedelta

class Bucket(object):
    def __init__(self, period):
        self.period_delta = timedelta(seconds=period)
        self.reset_time = datetime.now()
        self.quota = 0

    def __repr__(self):
        return 'Wiadro(quota=%d)' % self.quota

bucket = Bucket(60)
print(bucket)


# Przykład 2.
def fill(bucket, amount):
    now = datetime.now()
    if now - bucket.reset_time > bucket.period_delta:
        bucket.quota = 0
        bucket.reset_time = now
    bucket.quota += amount


# Przykład 3.
def deduct(bucket, amount):
    now = datetime.now()
    if now - bucket.reset_time > bucket.period_delta:
        return False
    if bucket.quota - amount < 0:
        return False
    bucket.quota -= amount
    return True


# Przykład 4.
bucket = Bucket(60)
fill(bucket, 100)
print(bucket)


# Przykład 5.
if deduct(bucket, 99):
    print('Potrzebne 99 jednostek danych')
else:
    print('Nie ma 99 jednostek danych')
print(bucket)


# Przykład 6.
if deduct(bucket, 3):
    print('Potrzebne 3 jednostki danych')
else:
    print('Nie ma 3 jednostek danych')
print(bucket)


# Przykład 7.
class Bucket(object):
    def __init__(self, period):
        self.period_delta = timedelta(seconds=period)
        self.reset_time = datetime.now()
        self.max_quota = 0
        self.quota_consumed = 0

    def __repr__(self):
        return ('Bucket(max_quota=%d, quota_consumed=%d)' %
                (self.max_quota, self.quota_consumed))


# Przykład 8.
    @property
    def quota(self):
        return self.max_quota - self.quota_consumed


# Przykład 9.
    @quota.setter
    def quota(self, amount):
        delta = self.max_quota - amount
        if amount == 0:
            # Liczba jednostek danych jest wyzerowana dla nowego przedziału czasu.
            self.quota_consumed = 0
            self.max_quota = 0
        elif delta < 0:
            # Liczba jednostek danych przygotowana dla nowego przedziału czasu.
            assert self.quota_consumed == 0
            self.max_quota = amount
        else:
            # Liczba jednostek danych wykorzystana w przedziale czasu.
            assert self.max_quota >= self.quota_consumed
            self.quota_consumed += delta


# Przykład 10.
bucket = Bucket(60)
print('Początkowo', bucket)
fill(bucket, 100)
print('Wypełniono', bucket)

if deduct(bucket, 99):
    print('Potrzebne 99 jednostek danych')
else:
    print('Nie ma 99 jednostek danych')

print('Teraz', bucket)

if deduct(bucket, 3):
    print('Potrzebne 3 jednostki danych')
else:
    print('Nie ma 3 jednostek danych')

print('Pozostało', bucket)
