"""Funkcje straty do zadania wykrywania obiektów

"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
from __future__ import unicode_literals

import tensorflow as tf
from tensorflow.keras import backend as K
from tensorflow.keras.losses import Huber

import numpy as np

def focal_loss_ce(y_true, y_pred):
    """Alternatywna funkcja straty CE (nie używana)
    """
    # only missing in this FL is y_pred clipping
    weight = (1 - y_pred)
    weight *= weight
    # alpha = 0.25
    weight *= 0.25
    return K.categorical_crossentropy(weight*y_true, y_pred)


def focal_loss_binary(y_true, y_pred):
    """Funkcja straty - binarna entropia krzyżowa
    """
    gamma = 2.0
    alpha = 0.25

    pt_1 = tf.where(tf.equal(y_true, 1),
                    y_pred,
                    tf.ones_like(y_pred))
    pt_0 = tf.where(tf.equal(y_true, 0),
                    y_pred,
                    tf.zeros_like(y_pred))

    epsilon = K.epsilon()
    # przycinamy by zapobiec NaN oraz Inf
    pt_1 = K.clip(pt_1, epsilon, 1. - epsilon)
    pt_0 = K.clip(pt_0, epsilon, 1. - epsilon)

    weight = alpha * K.pow(1. - pt_1, gamma)
    fl1 = -K.sum(weight * K.log(pt_1))
    weight = (1 - alpha) * K.pow(pt_0, gamma)
    fl0 = -K.sum(weight * K.log(1. - pt_0))

    return fl1 + fl0


def focal_loss_categorical(y_true, y_pred):
    """Funkcja straty: zogniskowana skategoryzowana entropia krzyżowa"""
    gamma = 2.0
    alpha = 0.25

    # skala dla upewnienia się, że suma prawdopodobieństw wynosi 1.0
    y_pred /= K.sum(y_pred, axis=-1, keepdims=True)

    # przycięcie przewidywanych wartości, by zapobiec pojawieniu się NaN oraz Inf
    epsilon = K.epsilon()
    y_pred = K.clip(y_pred, epsilon, 1. - epsilon)

    # obliczenie entropii krzyżowej
    cross_entropy = -y_true * K.log(y_pred)

    # obliczenie straty zogniskowanej
    weight = alpha * K.pow(1 - y_pred, gamma)
    cross_entropy *= weight

    return K.sum(cross_entropy, axis=-1)


def mask_offset(y_true, y_pred): 
    """Wstępne przetwarzanie danych referencyjnych i przewidywanych"""
    # pierwsze 4 są przesunięciami
    offset = y_true[..., 0:4]
    # ostatnie 4 są maską
    mask = y_true[..., 4:8]
    # Predykcja jest faktycznie duplikowana dla dopasowania
    # albo dostajemy pierwsze lub ostatnie 4 predykcje przesunięć i stosujemy maskę.
    pred = y_pred[..., 0:4]
    offset *= mask
    pred *= mask
    return offset, pred


def l1_loss(y_true, y_pred):
    """MAE lub funkcja straty L1
    """
    offset, pred = mask_offset(y_true, y_pred)
    # możemy użyć L1
    return K.mean(K.abs(pred - offset), axis=-1)


def smooth_l1_loss(y_true, y_pred):
    """Gładka funkcja straty L1 wykorzystuje funkcję straty Hubera z tensorflow
    """
    offset, pred = mask_offset(y_true, y_pred)
    # Funkcja straty Hubera jest przybliżeniem gładkiej L1
    return Huber()(offset, pred)
