"""Konstruowanie, uczenie i ocena modelu IIC
"""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function


from tensorflow.keras.layers import Input, Dense, Flatten
from tensorflow.keras.models import Model
from tensorflow.keras.optimizers import Adam
from tensorflow.keras.callbacks import LearningRateScheduler
from tensorflow.keras.utils import plot_model
from tensorflow.keras import backend as K
from tensorflow.keras.datasets import mnist

import numpy as np
import os
import argparse
import vgg

from data_generator import DataGenerator
from utils import unsupervised_labels, center_crop
from utils import AccuracyCallback, lr_schedule


class IIC:
    def __init__(self,
                 args,
                    backbone):
        """ Zawiera model kodera, funkcję straty,
            ładowanie zestawów danych, trenowanie i procedury oceny
            w celu implementacji nienadzorowanego grupowania
            IIC poprzez maksymalizację informacji wzajemnej

        Argumenty:
            args: Argumenty wiersza poleceń wskazujące 
                wybór rozmiaru próbki, liczby czół, 
                folderu do zapisania pliku wag, 
                nazwy pliku wag itp. 
            backbone(Model): Szkielet kodera IIC (np. VGG)
        """

        self.args = args
        self.backbone = backbone
        self._model = None
        self.train_gen = DataGenerator(args, siamese=True)
        self.n_labels = self.train_gen.n_labels
        self.build_model()
        self.load_eval_dataset()
        self.accuracy = 0

    def build_model(self):
        """Konstruowanie modelu IIC z n czołami
        """

        inputs = Input(shape=self.train_gen.input_shape, name='x')
        x = self.backbone(inputs)
        x = Flatten()(x)
        # number of output heads
        outputs = []
        for i in range(self.args.heads):
            name = "z_head%d" % i
            outputs.append(Dense(self.n_labels,
                                 activation='softmax',
                                 name=name)(x))
        self._model = Model(inputs, outputs, name='encoder')
        optimizer = Adam(lr=1e-3)
        self._model.compile(optimizer=optimizer, loss=self.mi_loss)
        self._model.summary()

    def mi_loss(self, y_true, y_pred):
        """Strata informacji wzajemnej obliczona na podstawie
        macierzy wspólnego rozkładu i rozkładów brzegowych

        Argumenty:
        y_true (tensor): nie używamy — jest to uczenie nienadzorowane 
        y_pred (tensor): stos predykcji softmax 
            dla bliźniaczych wektorów niejawnych Z i Z kreskowane
        """
        size = self.args.batch_size
        n_labels = y_pred.shape[-1]
        # dolna połowa to Z
        Z = y_pred[0: size, :]
        Z = K.expand_dims(Z, axis=2)
        # górna połowa to Z kreskowane 
        Zbar = y_pred[size: y_pred.shape[0], :]
        Zbar = K.expand_dims(Zbar, axis=1)
        # obliczenie wspólnego rozkładu (równania 13.13 i 13.14)
        P = K.batch_dot(Z, Zbar)
        P = K.sum(P, axis=0)
        # wymuszenie symetrycznego wspólnego rozkładu (równanie 13.15)
        P = (P + K.transpose(P)) / 2.0
        # normalizacja prawdopodobieństwa całkowitego do 1.0
        P = P / K.sum(P)
        # rozkłady brzegowe (równania 13.16 i 13.17)
        Pi = K.expand_dims(K.sum(P, axis=1), axis=1)
        Pj = K.expand_dims(K.sum(P, axis=0), axis=0)
        Pi = K.repeat_elements(Pi, rep=n_labels, axis=1)
        Pj = K.repeat_elements(Pj, rep=n_labels, axis=0)
        P = K.clip(P, K.epsilon(), np.finfo(float).max)
        Pi = K.clip(Pi, K.epsilon(), np.finfo(float).max)
        Pj = K.clip(Pj, K.epsilon(), np.finfo(float).max)
        # ujemna strata MI (równanie 13.18)
        neg_mi = K.sum((P * (K.log(Pi) + K.log(Pj) - K.log(P))))
        # Każde czoło ma wkład 1/n_czoła do całkowitej straty
        return neg_mi/self.args.heads


    def train(self):
        """ Funkcja uczenia używa generatora danych,
            obliczeń dokładności i współczynnika uczenia
            harmonogramu wywołań zwrotnych
        """
        accuracy = AccuracyCallback(self)
        lr_scheduler = LearningRateScheduler(lr_schedule,
                                             verbose=1)
        callbacks = [accuracy, lr_scheduler]
        self._model.fit(x=self.train_gen,
                        use_multiprocessing=False,
                        epochs=self.args.epochs,
                        callbacks=callbacks,
                        shuffle=True)


    def load_eval_dataset(self):
        """Wstępne załadowanie danych testowych do oceny
        """
        (_, _), (x_test, self.y_test) = self.args.dataset.load_data()
        image_size = x_test.shape[1]
        x_test = np.reshape(x_test,[-1, image_size, image_size, 1])
        x_test = x_test.astype('float32') / 255
        x_eval = np.zeros([x_test.shape[0], *self.train_gen.input_shape])
        for i in range(x_eval.shape[0]):
            x_eval[i] = center_crop(x_test[i])

        self.x_test = x_eval


    def load_weights(self):
        """Przeładuj wagi modelu do oceny
        """
        if self.args.restore_weights is None:
            raise ValueError("Wagi modelu musza być załadowane pred oceną")

        if self.args.restore_weights:
            folder = "weights"
            os.makedirs(folder, exist_ok=True) 
            path = os.path.join(folder, self.args.restore_weights)
            print("Ladowanie wag... ", path)
            self._model.load_weights(path)


    def eval(self):
        """Ocena dokładności wag bieżącego modelu
        """
        y_pred = self._model.predict(self.x_test)
        print("")
        # dokładność na czoło
        for head in range(self.args.heads):
            if self.args.heads == 1:
                y_head = y_pred
            else:
                y_head = y_pred[head]
            y_head = np.argmax(y_head, axis=1)

            accuracy = unsupervised_labels(list(self.y_test),
                                           list(y_head),
                                           self.n_labels,
                                           self.n_labels)
            info = "Czolo %d dokladnosc: %0.2f%%"
            if self.accuracy > 0:
                info += ", Najlepsza stara dokladnosc: %0.2f%%" 
                data = (head, accuracy, self.accuracy)
            else:
                data = (head, accuracy)
            print(info % data)
            # jeśli dokładność się polepsza podczas uczenia, 
            # zapisz wagi modelu do pliku
            if accuracy > self.accuracy \
                    and self.args.save_weights is not None:
                self.accuracy = accuracy
                folder = self.args.save_dir
                os.makedirs(folder, exist_ok=True) 
                path = os.path.join(folder, self.args.save_weights)
                print("Zapisywanie wag... ", path)
                self._model.save_weights(path)


    @property
    def model(self):
        return self._model


if __name__ == '__main__':

    parser = argparse.ArgumentParser(description='IIC Keras')
    parser.add_argument('--save-dir',
                       default="weights",
                       help='Folder do zachowywania wag modelu (h5)')
    parser.add_argument('--save-weights',
                       default=None,
                       help='Folder do zachowywania wag modelu  (h5)')
    parser.add_argument('--dataset',
                       default=mnist,
                       help='Zbior danych ktorego nalezy uzyc')
    parser.add_argument('--epochs',
                        type=int,
                        default=1200,
                        metavar='N',
                        help='Liczba epok treningu')
    parser.add_argument('--batch-size',
                        type=int,
                        default=512,
                        metavar='N',
                        help='Wielkosc partii uczacej')
    parser.add_argument('--heads',
                        type=int,
                        default=1,
                        metavar='N',
                        help='Liczba czol')
    parser.add_argument('--train',
                        default=False,
                        action='store_true',
                        help='Trenowanie modelu')
    parser.add_argument('--restore-weights',
                        default=None,
                        help='Przywrócenie zapisanych wag modelu')
    parser.add_argument('--eval',
                        default=False,
                        action='store_true',
                        help='Ocena wstępnie wytrenowanego modelu. Trzeba wskazać plik z wagami.')
    parser.add_argument('--crop',
                        type=int,
                        default=4,
                        help='Piksele do przycięcia z obrazu')
    parser.add_argument('--plot-model',
                        default=False,
                        action='store_true',
                        help='Wyrysuj wszystkie modele sieci')

    args = parser.parse_args()

    # build backbone
    backbone = vgg.VGG(vgg.cfg['F'])
    backbone.model.summary()
    # instantiate IIC object
    iic = IIC(args, backbone.model)
    if args.plot_model:
        plot_model(backbone.model,
                   to_file="model-vgg.png",
                   show_shapes=True)
        plot_model(iic.model,
                   to_file="model-iic.png",
                   show_shapes=True)
    if args.eval:
        iic.load_weights()
        iic.eval()
    elif args.train:
        iic.train()
