""" Kod implementujący różne strategie metod gradientowych 
    do rozwiązania problemu samochodu wjeżdżającego pod górę (wersja ciągła) 
    (MountainCarCountinuous-v0).

Zaimplementowane metody:
    1) WZMOCNIENIE
    2) WZMOCNIENIE z wartością bazową
    3) Aktor-Krytyk
    4) A2C

Źródła:
    1) Sutton and Barto, Reinforcement Learning: An Introduction
    (2017)

    2) Mnih, et al. Asynchronous Methods for Deep Reinforcement
    Learning. Intl Conf on Machine Learning. 2016

"""

from tensorflow.keras.layers import Dense, Input
from tensorflow.keras.layers import Lambda, Activation
from tensorflow.keras.models import Model
from tensorflow.keras.optimizers import Adam, RMSprop
from tensorflow.keras import backend as K
from tensorflow.keras.utils import get_custom_objects
from tensorflow.keras.utils import plot_model

import tensorflow as tf
import tensorflow_probability as tfp

import numpy as np
import argparse
import gym
from gym import wrappers, logger
import csv
import time
import os
import datetime
import math


def softplusk(x):
    """ Niektóre implementacje używają zmodyfikowanego softplus, aby upewnić się,
        że odchylenie standardowe nigdy się nie wyzeruje
    Argument:
        x (tensor): wejście aktywacji
    """
    return K.softplus(x) + 1e-10


class PolicyAgent:
    def __init__(self, env):
        """Implementacja modelu i uczenia 
            metod strategii gradientowych
        Argument:
            env (Object): środowisko OpenAI gym 
        """

        self.env = env
        # waga straty entropii
        self.beta = 0.0
        # strata wartości dla wszystkich strategii gradientowych 
        # z wyjątkiem A2C
        self.loss = self.value_loss
        
        # s, a, r, s' są przechowywane w pamięci
        self.memory = []

        # do obliczenia rozmiaru wejściowego
        self.state = env.reset()
        self.state_dim = env.observation_space.shape[0]
        self.state = np.reshape(self.state, [1, self.state_dim])
        self.build_autoencoder()


    def reset_memory(self):
        """Czyści pamięć przed rozpoczęciem każdego epizodu
        """
        self.memory = []


    def remember(self, item):
        """Zapamiętuje każdy s,a,r,s' w każdym kroku epizodu
        """
        self.memory.append(item)


    def action(self, args):
        """Mając: średnią, odchylenie standardowe, próbkę akcji, przycięcie i zysk, 
            zakładamy, że prawdopodobieństwo ma rozkład Gaussa,
            i wybieramy akcję przy zadanym stanie
        Argument:
            args (list): lista średnich i odchyleń standardowych
        Zwraca:
            action (tensor): strategię akcji
        """
        mean, stddev = args
        dist = tfp.distributions.Normal(loc=mean, scale=stddev)
        action = dist.sample(1)
        action = K.clip(action,
                        self.env.action_space.low[0],
                        self.env.action_space.high[0])
        return action


    def logp(self, args):
        """Mając średnią, odchylenie standardowe i akcję, oblicza prawdopodobieństwo logarytmiczne rozkładu Gaussa
        Argument:
            args (list): średnia, odchylenie standardowe, akcja, lista
        Zwraca:
            logp (tensor): logarytm akcji
        """
        mean, stddev, action = args
        dist = tfp.distributions.Normal(loc=mean, scale=stddev)
        logp = dist.log_prob(action)
        return logp


    def entropy(self, args):
        """Mając średnią i odchylenie standardowe, oblicza entropię rozkładu Gaussa
        Argument:
            args (list): lista: średnia, odchylenie standardowe
        Zwraca:
            entropy (tensor): entropia akcji
        """
        mean, stddev = args
        dist = tfp.distributions.Normal(loc=mean, scale=stddev)
        entropy = dist.entropy()
        return entropy


    def build_autoencoder(self):
        """Sieć autokodująca do konwersji stanów w cechy
        """
        # po pierwsze budujemy model kodera
        inputs = Input(shape=(self.state_dim, ), name='state')
        feature_size = 32
        x = Dense(256, activation='relu')(inputs)
        x = Dense(128, activation='relu')(x)
        feature = Dense(feature_size, name='feature_vector')(x)

        # instancja modelu kodera
        self.encoder = Model(inputs, feature, name='encoder')
        self.encoder.summary()
        plot_model(self.encoder,
                   to_file='encoder.png', 
                   show_shapes=True)

        # budujemy model dekodera
        feature_inputs = Input(shape=(feature_size,), 
                               name='decoder_input')
        x = Dense(128, activation='relu')(feature_inputs)
        x = Dense(256, activation='relu')(x)
        outputs = Dense(self.state_dim, activation='linear')(x)

        # instancja modelu dekodera
        self.decoder = Model(feature_inputs, 
                             outputs, 
                             name='decoder')
        self.decoder.summary()
        plot_model(self.decoder, 
                   to_file='decoder.png', 
                   show_shapes=True)

        # sieć autokodująca = koder+dekoder
        # instancja modelu autokodera
        self.autoencoder = Model(inputs, 
                                 self.decoder(self.encoder(inputs)),
                                 name='autoencoder')
        self.autoencoder.summary()
        plot_model(self.autoencoder, 
                   to_file='autoencoder.png', 
                   show_shapes=True)

        # funkcja straty błędu średniokwadratowego (MSE), optymalizator Adam 
        self.autoencoder.compile(loss='mse', optimizer='adam')


    def train_autoencoder(self, x_train, x_test):
        """Uczenie sieci autokodującej z użyciem losowo próbkowanych stanów z otoczenia
        Argumenty:
            x_train (tensor): zbiór uczący sieci autokodującej
            x_test (tensor): zbiór testowy sieci autokodującej
        """
        # uczenie sieci autokodującej

        batch_size = 32
        self.autoencoder.fit(x_train,
                             x_train,
                             validation_data=(x_test, x_test),
                             epochs=10,
                             batch_size=batch_size)


    def build_actor_critic(self):
        """Konstruowane są 4 modele, ale 3 modele współdzielą te same parametry, stąd uczenie jednego wystarcza dla pozostałych 
        3 modele, które współdzielą te same parametry, to akcja, logp oraz entropia. 
        Model entropii jest używany tylko przez A2C. 
        Każdy model ma taką samą strukturę MLP: Wejście(2)-koder-Wyjście(1).
        Funkcja aktywacji wyjścia zależy od charakteru wyjścia
        """

        inputs = Input(shape=(self.state_dim, ), name='state')
        self.encoder.trainable = False
        x = self.encoder(inputs)
        mean = Dense(1,
                     activation='linear',
                     kernel_initializer='zero',
                     name='mean')(x)
        stddev = Dense(1,
                       kernel_initializer='zero',
                       name='stddev')(x)
        # użycie softplusk pozwala na uniknięcie sytuacji, 
        # że odchylenie standardowe = 0
        stddev = Activation('softplusk', name='softplus')(stddev)
        action = Lambda(self.action,
                        output_shape=(1,),
                        name='action')([mean, stddev])
        self.actor_model = Model(inputs, action, name='action')
        self.actor_model.summary()
        plot_model(self.actor_model, 
                   to_file='actor_model.png', 
                   show_shapes=True)

        logp = Lambda(self.logp,
                      output_shape=(1,),
                      name='logp')([mean, stddev, action])
        self.logp_model = Model(inputs, logp, name='logp')
        self.logp_model.summary()
        plot_model(self.logp_model, 
                   to_file='logp_model.png', 
                   show_shapes=True)

        entropy = Lambda(self.entropy,
                         output_shape=(1,),
                         name='entropy')([mean, stddev])
        self.entropy_model = Model(inputs, entropy, name='entropy')
        self.entropy_model.summary()
        plot_model(self.entropy_model, 
                   to_file='entropy_model.png', 
                   show_shapes=True)

        value = Dense(1,
                      activation='linear',
                      kernel_initializer='zero',
                      name='value')(x)
        self.value_model = Model(inputs, value, name='value')
        self.value_model.summary()
        plot_model(self.value_model, 
                   to_file='value_model.png', 
                   show_shapes=True)


        # strata logp sieci strategii
        loss = self.logp_loss(self.get_entropy(self.state), 
                              beta=self.beta)
        optimizer = RMSprop(lr=1e-3)
        self.logp_model.compile(loss=loss, optimizer=optimizer)

        optimizer = Adam(lr=1e-3)
        self.value_model.compile(loss=self.loss, optimizer=optimizer)


    def logp_loss(self, entropy, beta=0.0):
        """strata logp, trzecia i czwarta zmienna (entropia oraz beta) są potrzebne dla A2C,
            więc mamy różne struktury funkcji strat
        Argumenty:
            entropy (tensor): strata entropii
            beta (float): waga straty entropii
        Zwraca:
            loss (tensor): obliczona strata
        """
        def loss(y_true, y_pred):
            return -K.mean((y_pred * y_true) \
                    + (beta * entropy), axis=-1)

        return loss


    def value_loss(self, y_true, y_pred):
        """Typowa struktura funkcji straty, akceptująca tylko 2 argumenty.
            Będzie użyta jako strata wartości dla wszystkich metod z wyjątkiem A2C.
        Argumenty:
            y_true (tensor): wartość odniesienia
            y_pred (tensor): wartość z predykcji
        Zwraca:
            loss (tensor): obliczona strata
        """
        loss = -K.mean(y_pred * y_true, axis=-1)
        return loss


    def save_weights(self, 
                     actor_weights, 
                     encoder_weights, 
                     value_weights=None):
        """Zapisuje wagi aktora, krytyka i kodera. Przydatne dla zachowania wytrenowanych modeli
    
        Argumenty:
            actor_weights (tensor): parametry sieci aktora
            encoder_weights (tensor): wagi kodera
            value_weights (tensor): wartości parametrów sieci
        """
        self.actor_model.save_weights(actor_weights)
        self.encoder.save_weights(encoder_weights)
        if value_weights is not None:
            self.value_model.save_weights(value_weights)


    def load_weights(self,
                     actor_weights, 
                     value_weights=None):
        """Ładuje wagi wytrenowanej sieci. Przydatne gdy chcemy po prostu 
            użyć już przygotowanej sieci
        Argumenty:
            actor_weights (string): nazwa pliku zawierającego wagi sieci aktora
            value_weights (string): nazwa pliku zawierającego wagi sieci
        """
        self.actor_model.load_weights(actor_weights)
        if value_weights is not None:
            self.value_model.load_weights(value_weights)

    
    def load_encoder_weights(self, encoder_weights):
        """Ładowanie wag wytrenowanego kodera. Przydatne gdy chcemy po prostu 
            użyć już przygotowanej sieci.
        Argumenty:
            encoder_weights (string): nazwa pliku zawierającego wagi sieci kodera
        """
        self.encoder.load_weights(encoder_weights)

    
    def act(self, state):
        """Wywołanie sieci strategii by wypróbować akcję
        Argument:
            state (tensor): stan środowiska
        Zwraca:
            act (tensor): akcja strategii
        """
        action = self.actor_model.predict(state)
        return action[0]


    def value(self, state):
        """Wywołanie sieci wartości do predykcji wartości stanu
        Argument:
            state (tensor): stan środowiska
        Zwraca:
            value (tensor): wartość stanu
        """
        value = self.value_model.predict(state)
        return value[0]


    def get_entropy(self, state):
        """Zwraca entropię rozkładu strategii
        Argument:
            state (tensor): stan środowiska
        Zwraca:
            entropy (tensor): entropia strategii 
        """
        entropy = self.entropy_model.predict(state)
        return entropy[0]


class REINFORCEAgent(PolicyAgent):
    def __init__(self, env):
        """Implementacja modelu i treningu 
            metody strategii gradientowej WZMOCNIENIE
        Argumenty:
            env (Object): środowisko OpenAI gym
        """
        super().__init__(env) 

    def train_by_episode(self):
         """Uczenie epizodami
          Przygotuj zbiór danych przed uczeniem krok po kroku
        """
        # poniższego kodu używają  
        # tylko WZMOCNIENIE oraz WZMOCNIENIE wartością bazową
        # przekształcenie nagród na zyski

        rewards = []
        gamma = 0.99
        for item in self.memory:
            [_, _, _, reward, _] = item
            rewards.append(reward)
        # rewards = np.array(self.memory)[:,3].tolist()

        # obliczenie zysku na krok
        # zysk jest sumą nagród od momentu t do końca epizodu
        # zysk jest wstawiany na listę zamiast nagrody

        for i in range(len(rewards)):
            reward = rewards[i:]
            horizon = len(reward)
            discount =  [math.pow(gamma, t) for t in range(horizon)]
            return_ = np.dot(reward, discount)
            self.memory[i][3] = return_

        # uczenie — każdy krok
        for item in self.memory:
            self.train(item, gamma=gamma)


    def train(self, item, gamma=1.0):
        """Główna procedura ucząca 
        Argumenty:
            item (list): jedna jednostka doświadczenia 
            gamma (float): współczynnik dyskontowy [0,1]
        """
        [step, state, next_state, reward, done] = item

        # trzeba zapisać stan, potrzebne do obliczeń entropii
        self.state = state

        discount_factor = gamma**step
        delta = reward
        
        # zastosowanie współczynnika dyskontowego zgodnie z zapisami w algorytmach
        # 10.1, 10.2 i 10.3
        discounted_delta = delta * discount_factor
        discounted_delta = np.reshape(discounted_delta, [-1, 1])
        verbose = 1 if done else 0

        # uczenie modelu logp (co implikuje również uczenie modelu aktora,
        # jako że współdzielą dokładnie ten sam zbiór parametrów)
        self.logp_model.fit(np.array(state),
                            discounted_delta,
                            batch_size=1,
                            epochs=1,
                            verbose=verbose)


class REINFORCEBaselineAgent(REINFORCEAgent):
    def __init__(self, env):
        """Implementacja modelu i uczenia strategii metody gradientowej WZMOCNIENIE z wartością bazową
        Argumenty:
            env (Object): środowisko OpenAI gym 
        """
        super().__init__(env) 


    def train(self, item, gamma=1.0):
        """Główna procedura ucząca 
        Argumenty:
            item (list): jedna jednostka doświadczenia
            gamma (float): współczynnik dyskontowy [0,1]
        """

        [step, state, next_state, reward, done] = item

        # trzeba zapisać, potrzebne do obliczeń entropii
        self.state = state

        discount_factor = gamma**step

        # wzmocnienie z wartością_bazową: delta = zysk–wartość
        delta = reward - self.value(state)[0] 

        # zastosowanie współczynnika dyskontowego zgodnie z zapisami w algorytmach
        # 10.1, 10.2 oraz 10.3
        discounted_delta = delta * discount_factor
        discounted_delta = np.reshape(discounted_delta, [-1, 1])
        verbose = 1 if done else 0

        # uczenie modelu logp (co implikuje również uczenie modelu aktora,
        # jako że współdzielą dokładnie ten sam zbiór parametrów)
        self.logp_model.fit(np.array(state),
                            discounted_delta,
                            batch_size=1,
                            epochs=1,
                            verbose=verbose)
                            
        # uczenie sieci wartości (krytyk)
        self.value_model.fit(np.array(state),
                             discounted_delta,
                             batch_size=1,
                             epochs=1,
                             verbose=verbose)

class A2CAgent(PolicyAgent):
    def __init__(self, env):
        """Implementacja modelu i procedury uczenia metody strategii gradientowej A2C
        Argumenty:
            env (Object): środowisko OpenAI gym
        """
        super().__init__(env) 
        # beta — waga entropii używanej w A2C
        self.beta = 0.9
        # ustawienie funkcji straty dla modelu wartości A2C na mse
        self.loss = 'mse'


    def train_by_episode(self, last_value=0):
         """Uczenie epizodami 
            Przygotuj zbiór danych przed uczeniem krok po kroku
        Argumenty:
            last_value (float): poprzednia predykcja sieci wartości
        """
        
        # implementacja uczenia A2C od ostatniego do pierwszego stanu
        # współczynnik dyskontowy
        gamma = 0.95
        r = last_value
        # pamięć jest odwiedzana w odwrotnej kolejności, niż pokazano
        # w algorytmie 10.4
        for item in self.memory[::-1]:
            [step, state, next_state, reward, done] = item
            # oblicz zysk
            r = reward + gamma*r
            item = [step, state, next_state, r, done]
            # uczenie na krok
            # nagroda a2c zostaje zignorowana
            self.train(item)


    def train(self, item, gamma=1.0):
        """Główna procedura uczenia
        Argumenty:
            item (list): jedna jednostka doświadczenia
            gamma (float): współczynnik dyskontowy [0,1]
        """
        [step, state, next_state, reward, done] = item

        # trzeba zapisać, potrzebne do obliczeń entropii
        self.state = state

        discount_factor = gamma**step

        # a2c: delta = zdyskontowana wartość nagrody
        delta = reward - self.value(state)[0] 

        discounted_delta = delta * discount_factor
        discounted_delta = np.reshape(discounted_delta, [-1, 1])
        verbose = 1 if done else 0

        # uczenie modelu logp (to implikuje również uczenie modelu aktora),
        # ponieważ współdzielą ten sam zbiór parametrów
        self.logp_model.fit(np.array(state),
                            discounted_delta,
                            batch_size=1,
                            epochs=1,
                            verbose=verbose)

        # w A2C docelowa wartość jest zyskiem (nagroda zamieniona na zysk
        # w funkcji train_by_episode)
        discounted_delta = reward
        discounted_delta = np.reshape(discounted_delta, [-1, 1])

        # Trenuj sieć wartości (krytyk)
        self.value_model.fit(np.array(state),
                             discounted_delta,
                             batch_size=1,
                             epochs=1,
                             verbose=verbose)


class ActorCriticAgent(PolicyAgent):
    def __init__(self, env):
        """Implementacja modelu i uczenia metody strategii gradientowej Aktor-Krytyk
        Argumenty:
            env (Object): środowisko OpenAI gym
        """

        super().__init__(env) 


    def train(self, item, gamma=1.0):
        """Główna procedura ucząca
        Argumenty:
            item (list): jedna jednostka doświadczenia
            gamma (float): współczynnik dyskontowy [0,1]
        """

        [step, state, next_state, reward, done] = item

        # trzeba zapisać stan, potrzebne do obliczeń entropii
        self.state = state

        discount_factor = gamma**step

        # Aktor-Krytyk: delta = nagroda–wartość 
        #       +następna zdyskontowana wartość
        delta = reward - self.value(state)[0] 

        # ponieważ ta funkcja jest wywoływana bezpośrednio w Aktor-Krytyk,
        # oceniamy tutaj funkcję wartości
        if not done:
            next_value = self.value(next_state)[0]
            # add  the discounted next value
            delta += gamma*next_value

        # zastosowanie współczynnika dyskontowego zgodnie z zapisami w algorytmach
        # 10.1, 10.2 oraz 10.3
        discounted_delta = delta * discount_factor
        discounted_delta = np.reshape(discounted_delta, [-1, 1])
        verbose = 1 if done else 0

        # uczenie modelu logp (co implikuje również uczenie modelu aktora),
        # jako że współdzielą dokładnie ten sam zbiór parametrów
        self.logp_model.fit(np.array(state),
                            discounted_delta,
                            batch_size=1,
                            epochs=1,
                            verbose=verbose)

        self.value_model.fit(np.array(state),
                             discounted_delta,
                             batch_size=1,
                             epochs=1,
                             verbose=verbose)



def setup_parser():
    parser = argparse.ArgumentParser(description=None)
    parser.add_argument('env_id',
                        nargs='?',
                        default='MountainCarContinuous-v0',
                        help='Wybierz środowisko do uruchomienia')
    parser.add_argument("-b",
                        "--baseline",
                        action='store_true',
                        help="WZMOCNIENIE z wartością bazową")
    parser.add_argument("-a",
                        "--actor-critic",
                        action='store_true',
                        help="Aktor-Krytyk")
    parser.add_argument("-c",
                        "--a2c",
                        action='store_true',
                        help="Zaawansowany-Aktor-Krytyk (A2C)")
    parser.add_argument("-r",
                        "--random",
                        action='store_true',
                        help="Losowa akcja strategii")
    parser.add_argument("-w",
                        "--actor-weights",
                        help="Zaladuj model z wstepnie wytrenowanymi wagami aktora")
    parser.add_argument("-y",
                        "--value-weights",
                        help="Zaladuj model z wstepnie wytrenowanymi wagami")
    parser.add_argument("-e",
                        "--encoder-weights",
                        help="Zaladuj model z wstepnie wytrenowanymi wagami kodera")
    parser.add_argument("-t",
                        "--train",
                        help="Uruchom trening",
                        action='store_true')
    args = parser.parse_args()
    return args


def setup_files(args):
    """Ustawienia umożliwiające przechowywanie logów z wyjść w osobnych katalogów.
    Argument:
        args: zdefiniowane przez użytkownika
    """
    postfix = 'reinforce'
    has_value_model = False
    if args.baseline:
        postfix = "reinforce-baseline"
        has_value_model = True
    elif args.actor_critic:
        postfix = "actor-critic"
        has_value_model = True
    elif args.a2c:
        postfix = "a2c"
        has_value_model = True
    elif args.random:
        postfix = "random"

    # create the folder for log files
    try:
        os.mkdir(postfix)
    except FileExistsError:
        print(postfix, "Katalog już istnieje")

    fileid = "%s-%d" % (postfix, int(time.time()))
    actor_weights = "actor_weights-%s.h5" % fileid
    actor_weights = os.path.join(postfix, actor_weights)
    encoder_weights = "encoder_weights-%s.h5" % fileid
    encoder_weights = os.path.join(postfix, encoder_weights)
    value_weights = None
    if has_value_model:
        value_weights = "value_weights-%s.h5" % fileid
        value_weights = os.path.join(postfix, value_weights)

    outdir = "/tmp/%s" % postfix

    misc = (postfix, fileid, outdir, has_value_model)
    weights = (actor_weights, encoder_weights, value_weights)

    return weights, misc



def setup_agent(env, args):
    """Inicjalizacja agenta
    Argumenty:
        env (Object): środowisko OpenAI
        args : argumenty zdefiniowane przez użytkownika
    """
    # instantiate agent
    if args.baseline:
        agent = REINFORCEBaselineAgent(env)
    elif args.a2c:
        agent = A2CAgent(env)
    elif args.actor_critic:
        agent = ActorCriticAgent(env)
    else:
        agent = REINFORCEAgent(env)

    # if weights are given, lets load them
    if args.encoder_weights:
        agent.load_encoder_weights(args.encoder_weights)
    else:
        x_train = [env.observation_space.sample() \
                   for x in range(200000)]
        x_train = np.array(x_train)
        x_test = [env.observation_space.sample() \
                  for x in range(20000)]
        x_test = np.array(x_test)
        agent.train_autoencoder(x_train, x_test)

    agent.build_actor_critic()
    train = True
    # if weights are given, lets load them
    if args.actor_weights:
        train = False
        if args.value_weights:
            agent.load_weights(args.actor_weights,
                               args.value_weights)
        else:
            agent.load_weights(args.actor_weights)

    return agent, train


def setup_writer(fileid, postfix):
    """Do przygotowanie plików i zapisu danych
    Argumenty:
        fileid (string): unikalny identyfikator pliku
        postfix (string): ścieżka do pliku
    """
    # we dump episode num, step, total reward, and 
    # number of episodes solved in a csv file for analysis
    csvfilename = "%s.csv" % fileid
    csvfilename = os.path.join(postfix, csvfilename)
    csvfile = open(csvfilename, 'w', 1)
    writer = csv.writer(csvfile,
                        delimiter=',',
                        quoting=csv.QUOTE_NONNUMERIC)
    writer.writerow(['Epizod',
                     'Krok',
                     'Nagroda',
                     'Liczba rozwiązanych epizodów'])

    return csvfile, writer


if __name__ == '__main__':
    args = setup_parser()
    logger.setLevel(logger.ERROR)

    weights, misc = setup_files(args)
    actor_weights, encoder_weights, value_weights = weights
    postfix, fileid, outdir, has_value_model = misc

    env = gym.make(args.env_id)
    env = wrappers.Monitor(env, directory=outdir, force=True)
    env.seed(0)
    
    # zarejestrowanie funkcji aktywacji softplusk. 
    # Na wszelki wypadek gdyby Czytelnik jej potrzebował. 
    get_custom_objects().update({'softplusk':Activation(softplusk)})
   
    agent, train = setup_agent(env, args)

    if args.train or train:
        train = True
        csvfile, writer = setup_writer(fileid, postfix)

    # liczba epizodów które uruchomimy w czasie uczenia 
    episode_count = 1000
    state_dim = env.observation_space.shape[0]
    n_solved = 0 
    # próbkowanie i dopasowanie
    for episode in range(episode_count):
        state = env.reset()
        # stanem jest samochód [pozycja, prędkość]
        state = np.reshape(state, [1, state_dim])
        # zresetuj wszystkie zmienne i pamięć 
        # przed rozpoczęciem każdego epizodu
        step = 0
        total_reward = 0
        done = False
        agent.reset_memory()
        start_time = datetime.datetime.now()
        while not done:
            # [min, max] akcja = [–1.0, 1.0]
            # dla wartości bazowej: losowy wybór akcji nie przesunie
            # samochodu tak, by osiągnął słupek flagi
            if args.random:
                action = env.action_space.sample()
            else:
                action = agent.act(state)
            env.render()
            # po wykonaniu akcji pobierz s', r, done
            next_state, reward, done, _ = env.step(action)
            next_state = np.reshape(next_state, [1, state_dim])
            # zapisujemy jednostkę doświadczenia w pamięci do celów szkolenia
            # Aktor-Krytyk tego nie potrzebuje, ale mimo to zachowujemy.
            item = [step, state, next_state, reward, done]
            agent.remember(item)

            if args.actor_critic and train:
                # Tylko Aktor-Krytyk przeprowadza trening online;
                # uczenie na każdym kroku, aż się zdarzy
                agent.train(item, gamma=0.99)
            elif not args.random and done and train:
                # dla WZMOCNIENIE, WZMOCNIENIE z wartością bazową oraz A2C
                # Czekamy najpierw na zakończenie epizodu, zanim uczymy sieć(-ci). 
                # Ostatnia wartość jest używana przez A2C
                if args.a2c:
                    v = 0 if reward > 0 else agent.value(next_state)[0]
                    agent.train_by_episode(last_value=v)
                else:
                    agent.train_by_episode()

            # skumulowana nagroda
            total_reward += reward
            # następnym stanem jest nowy stan
            state = next_state
            step += 1

        if reward > 0:
            n_solved += 1
        elapsed = datetime.datetime.now() - start_time
        fmt = "Episode=%d, Step=%d, Action=%f, Reward=%f"
        fmt = fmt + ", Total_Reward=%f, Elapsed=%s"
        msg = (episode, step, action[0], reward, total_reward, elapsed)
        print(fmt % msg)
        # zapis danych do otwartego pliku csv, do analiz
        if train:
            writer.writerow([episode, step, total_reward, n_solved])



    # po przeprowadzeniu treningu, zapis wartości wag modelu aktora oraz wartości
    if not args.random and train:
        if has_value_model:
            agent.save_weights(actor_weights,
                               encoder_weights,
                               value_weights)
        else:
            agent.save_weights(actor_weights,
                               encoder_weights)

    # zapisanie środowiska i zapis monitorowanych rezultatów na dysk
    if train:
        csvfile.close()
    env.close() 
