
"""Q-Learning do rozwiązania problemu zamarzniętego jeziora (FrozenLake-v0 problem)

"""

from collections import deque
import numpy as np
import argparse
import os
import time
import gym
from gym import wrappers, logger

class QAgent:
    def __init__(self,
                 observation_space,
                 action_space,
                 demo=False,
                 slippery=False,
                 episodes=40000):
        """Q-uczenie agenta w środowisku FrozenLake-v0

        Argumenty:
            observation_space (tensor): przestrzeń stanów
            action_space (tensor): przestrzeń akcji
            demo (Bool): czy tryb demo, czy uczenie
            slippery (Bool): dwie wersje środowiska FrozenLake-v0
            episodes (int): liczba epizodów podczas treningu
        """

        
        self.action_space = action_space
        # liczba kolumn jest równa liczbie akcji
        col = action_space.n
        # liczba wierszy jest równa liczbie stanów
        row = observation_space.n
        # tworzenie tabeli Q o wymiarach wiersze x kolumny
        self.q_table = np.zeros([row, col])

        # współczynnik dyskontowy
        self.gamma = 0.9

        # początkowo eksploracja 90%, eksploatacja 10%
        self.epsilon = 0.9
        # iteracyjne stosowanie zanikania aż do osiągnięcia 
        # 10% eksploracji/90% eksploatacji

        self.epsilon_min = 0.1
        self.epsilon_decay = self.epsilon_min / self.epsilon
        self.epsilon_decay = self.epsilon_decay ** \
                             (1. / float(episodes))

        # współczynnik uczenia Q-uczenia
        self.learning_rate = 0.1
        
        # plik, w którym jest zachowywana (lub z którego jest odczytywana) tabela Q
        if slippery:
            self.filename = 'q-frozenlake-slippery.npy'
        else:
            self.filename = 'q-frozenlake.npy'

        # tryb demo czy uczenia?
        self.demo = demo
        # jeśli demo, bez eksploracji
        if demo:
            self.epsilon = 0

    def act(self, state, is_explore=False):
        """określenie następnej akcji
            jeśli losowa, wybierz z losowej przestrzeni akcji;
            w innym wypadku wybierz z tabeli wartości Q
        Argumenty:
            state (tensor): aktualny stan agenta
            is_explore (Bool): czy tryb eksploracji?
        Zwraca:
            action (tensor): akcja, którą ma wykonać agent
        """
        # 0 — lewo, 1 — dół, 2 — prawo, 3 — góra

        if is_explore or np.random.rand() < self.epsilon:
            # eksploracja — wykonaj losową akcję
            return self.action_space.sample()

        # eksploatacja — wybierz akcje z maksymalną wartością Q
        action = np.argmax(self.q_table[state])
        return action


    def update_q_table(self, state, action, reward, next_state):
        """uczenie TD(0)(uogólnione Q-uczenie) ze współczynnikiem uczenia
        Argumenty:
            state (tensor): stan otoczenia
            action (tensor): akcja wykonana przez agenta dla danego stanu
            reward (float): nagroda, którą otrzyma agent za daną akcję
            next_state (tensor): następny stan otoczenia
        """
        # Q(s, a) +=
        # alpha * (reward+gamma * max_a' Q(s', a') — Q(s, a))

        q_value = self.gamma * np.amax(self.q_table[next_state])
        q_value += reward
        q_value -= self.q_table[state, action]
        q_value *= self.learning_rate
        q_value += self.q_table[state, action]
        self.q_table[state, action] = q_value


    def print_q_table(self):
        """zrzut tabeli Q"""
        print(self.q_table)
        print("Epsilon : ", self.epsilon)


    def save_q_table(self):
        """Zapis wytrenowanej Tabeli Q"""
        np.save(self.filename, self.q_table)


    def load_q_table(self):
        """Załadowanie wytrenowanej Tabeli Q"""
        self.q_table = np.load(self.filename)


    def update_epsilon(self):
        """dostosowanie epsilon"""
        if self.epsilon > self.epsilon_min:
            self.epsilon *= self.epsilon_decay


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description=None)
    parser.add_argument('env_id',
                        nargs='?',
                        default='FrozenLake-v0',
                        help='Wybierz środowisko do uruchomienia')
    help_ = "Demo - wytrenowana Tabela Q"
    parser.add_argument("-d",
                        "--demo",
                        help=help_,
                        action='store_true')
    help_ = "Zamarznięte jezioro jest śliskie"
    parser.add_argument("-s",
                        "--slippery",
                        help=help_,
                        action='store_true')
    help_ = "Tylko eksploracja. Początek."
    parser.add_argument("-e",
                        "--explore",
                        help=help_,
                        action='store_true')
    help_ = "Sekundy opóźnienia w interfejsie użytkownika. Użyteczne podczas wizualizacji w trybie demo."
    parser.add_argument("-t",
                        "--delay",
                        help=help_,
                        type=int)
    args = parser.parse_args()

    logger.setLevel(logger.INFO)

    # instancja środowiska gym (FrozenLake-v0)
    env = gym.make(args.env_id)

    # katalog informacje do debugowania
    outdir = "/tmp/q-learning-%s" % args.env_id
    env = wrappers.Monitor(env, directory=outdir, force=True)
    env.seed(0)
    if not args.slippery:
        env.is_slippery = False

    if args.delay is not None:
        delay = args.delay 
    else: 
        delay = 0

    # ile razy osiągnięto stan "Cel"
    wins = 0
    # Liczba epizodów trwania treningu
    episodes = 40000

    # instancja agenta w środowisku Q-uczenia
    agent = QAgent(env.observation_space,
                   env.action_space,
                   demo=args.demo,
                   slippery=args.slippery,
                   episodes=episodes)

    if args.demo:
        agent.load_q_table()

   # pętla przez założoną liczbę epizodów
    for episode in range(episodes):
        state = env.reset()
        done = False
        while not done:
            # określenie akcji dla agenta w danym stanie 
            action = agent.act(state, is_explore=args.explore)
            # pobierz obserwowalne dane
            next_state, reward, done, _ = env.step(action)
            # wyczyść ekran przed renderowaniem środowiska
            os.system('clear')
            # wyrenderuj środowisko do debugowania przez człowieka
            env.render()
            # uczenie tabeli wartości Q
            if done:
                # aktualizacja proporcji eksploracja/eksploatacja
                # nagroda > 0, tylko jeśli osiągnięty Cel;
                # w innym wypadku to Dziura

                if reward > 0:
                    wins += 1

            if not args.demo:
                agent.update_q_table(state,
                                     action, 
                                     reward, 
                                     next_state)
                agent.update_epsilon()

            state = next_state
            percent_wins = 100.0 * wins / (episode + 1)
            print("-------%0.2f%% Goals in %d Episodes---------"
                  % (percent_wins, episode))
            if done:
                time.sleep(5 * delay)
            else:
                time.sleep(delay)


    print("Episodes: ", episode)
    print("Goals/Holes: %d/%d" % (wins, episode - wins))

    agent.print_q_table()
    if not args.demo and not args.explore:
        agent.save_q_table()
    # zamknij środowisko i zapisz wyniki monitorowania na dysk
    env.close() 
