"""
Peny kod do gry connect 4 na planszy board_width, board_height i dugoci wygranej, ktr mona okreli za pomoc odpowiednich 
metod. Pozwala na gr w connect 5, 6, 7, itd. Domylnie board_width = 7, board_height = 6, winning_length = 4

Gwn metod w tym kodzie jest play_game, ktra symuluje gr. Gra toczy si do koca na podstawie przekazanych argumentw, ktre pozwalaj okreli 
ruchy graczy.
Plansza jest reprezentowana przez krotk liczb cakowitych board_width x board_height. 0 oznacza, e aden gracz nie zagra w polu, 1 oznacza 
e zagra tam gracz numer 1, -1 oznacza, e zagra tam drugi gracz. Do zwrcenia kopii okrelonego stanu w wyniku wykonania ruchu mona uy metody apply_move
. Moe to by przydatne przy prbkowaniu min-max lub Monte Carlo.
"""

import random


def _new_board(board_width, board_height):
    """Zwraca pust plansz, ktrej moemy uy do symulacji gry.

    Argumenty:
        board_width (int): Szeroko planszy. Tworzona jest plansza o wymiarach board_width * board_height
        board_height (int): Wysoko planszy. Tworzona jest plansza o wymiarach board_width * board_height

    Funkcja zwraca:
        krotk liczb cakowitych board_width x board_height
    """
    return tuple(tuple(0 for _ in range(board_height)) for _ in range(board_width))


def apply_move(board_state, move_x, side):
    """Zwraca kopi danego stanu board_state po zastosowaniu danego ruchu.

    Argumenty:
        board_state (dwuelementowa krotka liczb cakowitych): Podany board_state, dla ktrego chcemy wykona ruch.
        move_x (int): W ktrej kolumnie chcemy "umieci" pion
        side (int): Strona, dla ktrej wykonujemy ten ruch, 1 dla pierwszego gracza, -1 dla drugiego gracza.

    Funkcja zwraca:
        (dwuwymiarowa krotka liczb cakowitych): Kopia stanu board_state z danym ruchem zastosowanym w imieniu danej strony.
    """
    # znajd pozycj dla ruchu 
    move_y = 0
    for x in board_state[move_x]:
        if x == 0:
            break
        else:
            move_y += 1

    def get_tuples():
        for i in range(len(board_state)):
            if move_x == i:
                temp = list(board_state[i])
                temp[move_y] = side
                yield tuple(temp)
            else:
                yield board_state[i]

    return tuple(get_tuples())


def available_moves(board_state):
    """Pobiera wszystkie prawidowe ruchy dla biecego stanu board_state. Dla gry w kko i krzyyk s to wszystkie pozycje, ktre aktualnie nie maj
    ruchu.

    Argumenty:
        board_state: stan board_state, dla ktrego chcemy sprawdzi prawidowe ruchy.

    Funkcja zwraca:
        Generator liczb int: Wszystkie poprawne ruchy, ktre mog by wykonane w tej pozycji.
    """
    for x in range(len(board_state)):
        if any(y == 0 for y in board_state[x]):
            yield x


def _has_winning_line(line, winning_length):
    count = 0
    last_side = 0
    for x in line:
        if x == last_side:
            count += 1
            if count == winning_length:
                return last_side
        else:
            count = 1
            last_side = x
    return 0


def has_winner(board_state, winning_length=4):
    """Okrela, czy gracz wygra dla danego stanu board_state.

    Argumenty:
        board_state (dwuwymiarowa krotka liczb cakowitych): Biecy stan board_state, ktry chcemy oceni.
        winning_length (int): Liczba ruchw w wierszu potrzebnych do wygranej.

    Funkcja zwraca:
        int: 1, jeli wygra gracz 1, -1, jeli wygra gracz 2, w przeciwnym razie 0.
    """
    board_width = len(board_state)
    board_height = len(board_state[0])

    # sprawdzanie wierszy
    for x in range(board_width):
        winner = _has_winning_line(board_state[x], winning_length)
        if winner != 0:
            return winner
    # sprawdzanie kolumn
    for y in range(board_height):
        winner = _has_winning_line((i[y] for i in board_state), winning_length)
        if winner != 0:
            return winner

    # sprawdzanie przektnych
    diagonals_start = -(board_width - winning_length)
    diagonals_end = (board_width - winning_length)
    for d in range(diagonals_start, diagonals_end):
        winner = _has_winning_line(
            (board_state[i][i + d] for i in range(max(-d, 0), min(board_width, board_height - d))),
            winning_length)
        if winner != 0:
            return winner
    for d in range(diagonals_start, diagonals_end):
        winner = _has_winning_line(
            (board_state[i][board_height - i - d - 1] for i in range(max(-d, 0), min(board_width, board_height - d))),
            winning_length)
        if winner != 0:
            return winner

    return 0  # nikt nie wygra, zwracamy 0 oznaczajce remis 


def play_game(plus_player_func, minus_player_func, board_width=7, board_height=6, winning_length=4, log=False):
    """Uruchamia pojedyncz gr w kko i krzyyk, a do koca dla podanych argumentw funkcji args, w celu okrelenia ruchw dla kadego 
    gracza.

    Argumenty:
        plus_player_func ((board_state(board_size by board_size tuple of int), side(int)) -> move((int, int))):
            Funkcja, ktra pobiera aktualny stan board_state i stron, po ktrej gra gracz i zwraca ruch gracza
            .
        minus_player_func ((board_state(board_size by board_size tuple of int), side(int)) -> move((int, int))):
            Funkcja, ktra pobiera aktualny stan board_state i stron, po ktrej gra gracz i zwraca ruch gracza
            .
        board_width (int): Szeroko planszy. Tworzona jest plansza o wymiarach board_width * board_height
        board_height (int): Wysoko planszy. Tworzona jest plansza o wymiarach board_width * board_height
        winning_length (int): Liczba ruchw w wierszu potrzebnych do wygranej.
        log (bool): Jeli True postpy s rejestrowane na konsoli, domylnie False 

    Funkcja zwraca:
        int: 1, jeli wygraa funkcja plus_player_func, -1 jeli wygraa funkcja minus_player_func oraz 0 w przypadku remisu
    """
    board_state = _new_board(board_width, board_height)
    player_turn = 1

    while True:
        _avialable_moves = list(available_moves(board_state))
        if len(_avialable_moves) == 0:
            # remis
            if log:
                print("brak dostpnych ruchw, gra zakoczya si remisem")
            return 0.
        if player_turn > 0:
            move = plus_player_func(board_state, 1)
        else:
            move = minus_player_func(board_state, -1)

        if move not in _avialable_moves:
            # jeli gracz wykonuje nieprawidowy ruch, drugi gracz wygrywa
            if log:
                print("niedozwolony ruch ", move)
            return -player_turn

        board_state = apply_move(board_state, move, player_turn)
        if log:
            print(board_state)

        winner = has_winner(board_state, winning_length)
        if winner != 0:
            if log:
                print("mamy zwycizce, strona: %s" % player_turn)
            return winner
        player_turn = -player_turn


def random_player(board_state, _):
    """Funkcja gracza, ktrej mona uy w metodzie play_game. Na podstawie stanu planszy, wybiera losowo ruch z 
    zbioru prawidowych ruchw w biecym stanie.

    Argumenty:
        board_state (dwuelementowa krotka liczb cakowitych): Aktualny stan planszy 
        _: strona, po ktrej gra gracz, nie jest uywana w tej funkcji, poniewa wybieramy ruchy losowo

    Funkcja zwraca:
        (int, int): ruch, ktry chcemy zagra na biecej planszy
    """
    moves = list(available_moves(board_state))
    return random.choice(moves)


if __name__ == '__main__':
    # przykad gry 
    play_game(random_player, random_player, log=True, board_width=7, board_height=6, winning_length=4)
