"""
Peny kod do gry w kko i krzyyk na planszy 3 na 3.
Dwaj gracze na zmian wykonuj ruchy na polach planszy. Pierwszy z nich, ktry uzyska 3 ruchy w wierszu, wcznie z  przektnymi wygrywa. Jeli
nie mona wykona adnych prawidowych ruchw, gra koczy si remisem.

Gwn metod w tym kodzie jest play_game, ktra symuluje gr. Gra toczy si do koca na podstawie przekazanych argumentw, ktre pozwalaj okreli 
ruchy graczy.
Plansza jest reprezentowana przez krotk liczb cakowitych o wymiarach 3 x 3. 0 oznacza, e aden gracz nie zagra w polu, 1 oznacza 
e zagra tam gracz numer 1, -1 oznacza, e zagra tam drugi gracz. Do zwrcenia kopii okrelonego stanu w wyniku wykonania ruchu mona uy metody apply_move
. Moe to by przydatne przy prbkowaniu min-max lub Monte Carlo.
"""
import random
import itertools


def _new_board():
    """Zwraca pust tablic gry w kko i krzyyk, ktrej moemy uy do symulacji gry.

    Funkcja zwraca:
        krotk liczb cakowitych o wymiarach 3x3
    """
    return ((0, 0, 0),
            (0, 0, 0),
            (0, 0, 0))


def apply_move(board_state, move, side):
    """Zwraca kopi danego stanu board_state po zastosowaniu danego ruchu.

    Argumenty:
        board_state (krotka liczb cakowitych o wymiarach 3x3): Podany board_state, dla ktrego chcemy wykona ruch.
        move (int, int): Pozycja, na ktrej chcemy wykona ruch.
        side (int): Strona, dla ktrej wykonujemy ten ruch, 1 dla pierwszego gracza, -1 dla drugiego gracza.

    Funkcja zwraca:
        (krotk liczb cakowitych o wymiarach 3x3): Kopia stanu board_state z danym ruchem zastosowanym w imieniu danej strony.
    """
    move_x, move_y = move

    def get_tuples():
        for x in range(3):
            if move_x == x:
                temp = list(board_state[x])
                temp[move_y] = side
                yield tuple(temp)
            else:
                yield board_state[x]

    return tuple(get_tuples())


def available_moves(board_state):
    """Pobiera wszystkie prawidowe ruchy dla biecego stanu board_state. Dla kko i krzyyk to wszystkie pozycje, ktre aktualnie nie maj ruchu
    .

    Argumenty:
        board_state: stan board_state, dla ktrego chcemy sprawdzi prawidowe ruchy.

    Funkcja zwraca:
        Generator krotek (int, int): Wszystkie poprawne ruchy, ktre mog by wykonane w tej pozycji.
    """
    for x, y in itertools.product(range(3), range(3)):
        if board_state[x][y] == 0:
            yield (x, y)


def _has_3_in_a_line(line):
    return all(x == -1 for x in line) | all(x == 1 for x in line)


def has_winner(board_state):
    """Okrela, czy gracz wygra dla danego stanu board_state.

    Argumenty:
        board_state (krotka liczb cakowitych o wymiarach 3x3): Biecy stan board_state, ktry chcemy oceni.

    Funkcja zwraca:
        int: 1, jeli wygra gracz 1, -1, jeli wygra gracz 2, w przeciwnym razie 0.
    """
    # sprawdzanie wierszy
    for x in range(3):
        if _has_3_in_a_line(board_state[x]):
            return board_state[x][0]
    # sprawdzanie kolumn
    for y in range(3):
        if _has_3_in_a_line([i[y] for i in board_state]):
            return board_state[0][y]

    # sprawdzanie przektnych
    if _has_3_in_a_line([board_state[i][i] for i in range(3)]):
        return board_state[0][0]
    if _has_3_in_a_line([board_state[2 - i][i] for i in range(3)]):
        return board_state[0][2]

    return 0  # nikt nie wygra, zwracamy 0 oznaczajce remis 


def play_game(plus_player_func, minus_player_func, log=False):
    """Uruchamia pojedyncz gr w kko i krzyyk, a do koca dla podanych argumentw funkcji args, w celu okrelenia ruchw dla kadego 
    gracza.

    Argumenty:
        plus_player_func ((board_state(3 by 3 tuple of int), side(int)) -> move((int, int))): Funkcja, ktra pobiera
            biecy board_state i stron, po ktrej gra gracz i zwraca ruch, jak gracz ma zagra.
        minus_player_func ((board_state(3 by 3 tuple of int), side(int)) -> move((int, int))): Funkcja, ktra pobiera
            biecy board_state i stron, po ktrej gra gracz i zwraca ruch, jak gracz ma zagra.
        log (bool): Jeli True postpy s rejestrowane na konsoli, domylnie False 

    Funkcja zwraca:
        int: 1, jeli wygraa funkcja plus_player_func, -1 jeli wygraa funkcja minus_player_func oraz 0 w przypadku remisu
    """
    board_state = _new_board()
    player_turn = 1

    while True:
        _available_moves = list(available_moves(board_state))

        if len(_available_moves) == 0:
            # remis
            if log:
                print("brak dostpnych ruchw, gra zakoczya si remisem")
            return 0.
        if player_turn > 0:
            move = plus_player_func(board_state, 1)
        else:
            move = minus_player_func(board_state, -1)

        if move not in _available_moves:
            # jeli gracz wykonuje nieprawidowy ruch, drugi gracz wygrywa
            if log:
                print("niedozwolony ruch ", move)
            return -player_turn

        board_state = apply_move(board_state, move, player_turn)
        if log:
            print(board_state)

        winner = has_winner(board_state)
        if winner != 0:
            if log:
                print("mamy zwycizce, strona: %s" % player_turn)
            return winner
        player_turn = -player_turn


def random_player(board_state, _):
    """Funkcja gracza, ktrej mona uy w metodzie play_game. Na podstawie stanu planszy, wybiera losowo ruch ze 
    zbioru prawidowych ruchw w biecym stanie.

    Argumenty:
        board_state (krotka liczb cakowitych o wymiarach 3x3): Aktualny stan planszy 
        _: strona, po ktrej gra gracz, nie jest uywana w tej funkcji, poniewa wybieramy ruchy losowo

    Funkcja zwraca:
        (int, int): ruch, ktry chcemy zagra na biecej planszy
    """
    moves = list(available_moves(board_state))
    return random.choice(moves)


if __name__ == '__main__':
    # przykad gry 
    play_game(random_player, random_player, log=True)
