"""
Peny kod do gry w kko i krzyyk na tablicy o dowolnym rozmiarze z okrelon liczb w wierszu jako wygranej. Jest to 
gra podobna do tic_tac_toe.py, ale wszystkie ruchy s sparametryzowane przez argument board_size, ktry okrela wielko planszy oraz 
winning_length ktry okrela, ile elementw w wierszu jest potrzebnych do wygranej. Wartoci domylne to 5 i 4. Dziki temu mona gra w gry 
w bardziej zoonym rodowisku ni standardowe kko i krzyyk.

Dwaj gracze na zmian wykonuj ruchy na polach planszy. Pierwszy z nich, ktry uzyska winning_length w wierszu, wcznie z  
przektnymi, wygrywa. Jeli nie mona wykona adnych prawidowych ruchw, gra koczy si remisem.

Gwn metod w tym kodzie jest play_game, ktra symuluje gr. Gra toczy si do koca na podstawie przekazanych argumentw, ktre pozwalaj okreli 
ruchy graczy.
Plansza jest reprezentowana przez krotk liczb cakowitych board_size na board_size. 0 oznacza, e aden gracz nie zagra w polu, 1 oznacza 
e zagra tam gracz numer 1, -1 oznacza, e zagra tam drugi gracz. Do zwrcenia kopii okrelonego stanu w wyniku wykonania ruchu mona uy metody apply_move
. Moe to by przydatne przy prbkowaniu min-max lub Monte Carlo.
"""
import random
import itertools


def _new_board(board_size):
    """Zwraca pust tablic gry w kko i krzyyk, ktrej moemy uy do symulacji gry.

    Argumenty:
        board_size (int): Rozmiar jednej strony planszy. Tworzy si plansza o wymiarach board_size * board_size 

    Funkcja zwraca:
        krotk liczb cakowitych board_size x board_size
    """
    return tuple(tuple(0 for _ in range(board_size)) for _ in range(board_size))


def apply_move(board_state, move, side):
    """Zwraca kopi danego stanu board_state po zastosowaniu danego ruchu.

    Argumenty:
        board_state (dwuwymiarowa krotka liczb cakowitych: Podany board_state, dla ktrego chcemy wykona ruch.
        move (int, int): Pozycja, na ktrej chcemy wykona ruch.
        side (int): Strona, dla ktrej wykonujemy ten ruch, 1 dla pierwszego gracza, -1 dla drugiego gracza.

    Funkcja zwraca:
        (dwuwymiarow krotk liczb cakowitych): Kopia stanu board_state z danym ruchem zastosowanym w imieniu danej strony.
    """
    move_x, move_y = move

    def get_tuples():
        for x in range(len(board_state)):
            if move_x == x:
                temp = list(board_state[x])
                temp[move_y] = side
                yield tuple(temp)
            else:
                yield board_state[x]

    return tuple(get_tuples())


def available_moves(board_state):
    """Pobiera wszystkie prawidowe ruchy dla biecego stanu board_state. Dla kko i krzyyk to wszystkie pozycje, w ktrych aktualnie nie wykonano ruchu
    .

    Argumenty:
        board_state: stan board_state, dla ktrego chcemy sprawdzi prawidowe ruchy.

    Funkcja zwraca:
        Generator krotek (int, int): Wszystkie poprawne ruchy, ktre mog by wykonane w tej pozycji.
    """
    for x, y in itertools.product(range(len(board_state)), range(len(board_state[0]))):
        if board_state[x][y] == 0:
            yield (x, y)


def _has_winning_line(line, winning_length):
    count = 0
    last_side = 0
    for x in line:
        if x == last_side:
            count += 1
            if count == winning_length:
                return last_side
        else:
            count = 1
            last_side = x
    return 0


def has_winner(board_state, winning_length):
    """Okrela, czy gracz wygra dla danego stanu board_state.

    Argumenty:
        board_state (dwuwymiarowa krotka liczb cakowitych): Biecy stan board_state, ktry chcemy oceni.
        winning_length (int): Liczba ruchw w wierszu potrzebnych do wygranej.

    Funkcja zwraca:
        int: 1, jeli wygra gracz 1, -1, jeli wygra gracz 2, w przeciwnym razie 0.
    """
    board_width = len(board_state)
    board_height = len(board_state[0])

    # sprawdzanie wierszy
    for x in range(board_width):
        winner = _has_winning_line(board_state[x], winning_length)
        if winner != 0:
            return winner
    # sprawdzanie kolumn
    for y in range(board_height):
        winner = _has_winning_line((i[y] for i in board_state), winning_length)
        if winner != 0:
            return winner

    # sprawdzanie przektnych
    diagonals_start = -(board_width - winning_length)
    diagonals_end = (board_width - winning_length)
    for d in range(diagonals_start, diagonals_end+1):
        winner = _has_winning_line(
            (board_state[i][i + d] for i in range(max(-d, 0), min(board_width, board_height - d))),
            winning_length)
        if winner != 0:
            return winner
    for d in range(diagonals_start, diagonals_end+1):
        winner = _has_winning_line(
            (board_state[i][board_height - i - d - 1] for i in range(max(-d, 0), min(board_width, board_height - d))),
            winning_length)
        if winner != 0:
            return winner

    return 0  # nikt nie wygra, zwracamy 0 oznaczajce remis 


def play_game(plus_player_func, minus_player_func, board_size=5, winning_length=4, log=False):
    """Uruchamia pojedyncz gr w kko i krzyyk, a do koca dla podanych argumentw funkcji args, w celu okrelenia ruchw dla kadego 
    gracza.

    Argumenty:
        plus_player_func ((board_state(board_size by board_size tuple of int), side(int)) -> move((int, int))):
            Funkcja, ktra pobiera aktualny stan board_state i stron, po ktrej gra gracz i zwraca ruch gracza
            .
        minus_player_func ((board_state(board_size by board_size tuple of int), side(int)) -> move((int, int))):
            Funkcja, ktra pobiera aktualny stan board_state i stron, po ktrej gra gracz i zwraca ruch gracza
            .
        board_size (int): Rozmiar pojedynczej strony planszy. Gra jest rozgrywana na planszy wielkoci board_size * board_size         winning_length (int): Liczba ruchw w wierszu potrzebnych do wygranej.
        log (bool): Jeli True postpy s rejestrowane na konsoli, domylnie False 
    Funkcja zwraca:
        int: 1, jeli wygraa funkcja plus_player_func, -1 jeli wygraa funkcja minus_player_func oraz 0 w przypadku remisu
    """
    board_state = _new_board(board_size)
    player_turn = 1

    while True:
        _available_moves = list(available_moves(board_state))
        if len(_available_moves) == 0:
            # remis
            if log:
                print("brak dostpnych ruchw, gra zakoczya si remisem")
            return 0.
        if player_turn > 0:
            move = plus_player_func(board_state, 1)
        else:
            move = minus_player_func(board_state, -1)

        if move not in _available_moves:
            # jeli gracz wykonuje nieprawidowy ruch, drugi gracz wygrywa
            if log:
                print("niedozwolony ruch ", move)
            return -player_turn

        board_state = apply_move(board_state, move, player_turn)
        print(board_state)

        winner = has_winner(board_state, winning_length)
        if winner != 0:
            if log:
                print("mamy zwycizce, strona: %s" % player_turn)
            return winner
        player_turn = -player_turn


def random_player(board_state, _):
    """Funkcja gracza, ktrej mona uy w metodzie play_game. Na podstawie stanu planszy, wybiera losowo ruch ze 
    zbioru prawidowych ruchw w biecym stanie.

    Argumenty:
        board_state (dwuwymiarowa krotka liczb cakowitych): Aktualny stan planszy 
        _: strona, po ktrej gra gracz, nie jest uywana w tej funkcji, poniewa wybieramy ruchy losowo

    Funkcja zwraca:
        (int, int): ruch, ktry chcemy zagra na biecej planszy
    """
    moves = list(available_moves(board_state))
    return random.choice(moves)


if __name__ == '__main__':
    # przykad gry 
    play_game(random_player, random_player, log=True, board_size=10, winning_length=4)
