library(shiny)
library(dplyr)
library(leaflet)

# Wczytywanie przygotowanych danych.
shootings <- read.csv("data/police-shootings.csv", stringsAsFactors = FALSE)

# Definiowanie interfejsu uytkownika aplikacji; interfejs wywietla map i tabel.
my_ui <- fluidPage(
  # Tytuł aplikacji.
  titlePanel("miertelne postrzelenia przez policj"),
  # Ramka boczna z widżetem selectInput do wyboru analizowanej zmiennej.
  sidebarLayout(
    sidebarPanel(
      selectInput(
        inputId = "analysis_var",
        label = "Poziom analiz",
        choices = c("gender", "race", "body_camera", "threat_level")
      )
    ),
    # Wyświetlanie mapy i tabeli w panelu głównym.
    mainPanel(
      leafletOutput("shooting_map"), # Dynamiczny element udostępniany przez pakiet leaflet.
      tableOutput("grouped_table")
    )
  )
)

# Definiowanie serwera wyświetlającego mapę i tabelę.
my_server <- function(input, output) {
  # Definiowanie mapy wyświetlanej w interfejsie użytkownika.
  output$shooting_map <- renderLeaflet({
    # Tworzenie palety kolorów (skali) opartej na wybranej analizowanej zmiennej.
    palette_fn <- colorFactor(palette = "Dark2", domain = shootings[[input$analysis_var]])
    
    # Tworzenie i zwracanie mapy.
    leaflet(data = shootings) %>%
      addProviderTiles("Stamen.TonerLite") %>% # Dodawanie kafelków mapy Stamen.
      addCircleMarkers( # Dodawanie znaczników do każdej strzelaniny.
        lat = ~lat,
        lng = ~long,
        label = ~paste0(name, ", ", age), # Dodawanie etykiety z nazwiskiem i wiekiem.
        color = ~palette_fn(shootings[[input$analysis_var]]), # Ustawianie koloru na podstawie danych wejściowych.
        fillOpacity = .7,
        radius = 4,
        stroke = FALSE
      ) %>%
      addLegend( # Dodawanie legendy do wykresu.
        "bottomright",
        title = "Rasa",
        pal = palette_fn, # Paleta kolorów punktów.
        values = shootings[[input$analysis_var]], # Notacja z podwójnym nawiasem kwadratowym.
        opacity = 1 # Legenda jest nieprzezroczysta.
      )
  })
  
  # Definiowanie tabeli wyświetlanej w interfejsie użytkownika.
  output$grouped_table <- renderTable({
    table <- shootings %>%
      group_by(shootings[[input$analysis_var]]) %>%
      count() %>%
      arrange(-n)
    colnames(table) <- c(input$analysis_var, "Liczba ofiar") # Formatowanie nazw kolumn.
    table # Zwracanie tabeli.
  })
}

# Uruchamianie aplikacji.
shinyApp(ui = my_ui, server = my_server)