# `tidyr` w praktyce: eksplorowanie statystyk z obszaru edukacji

# Wczytywanie potrzebnych pakietw
library(tidyr) # Na potrzeby przeksztacania danych
library(dplyr) # Na potrzeby przeksztacania danych
library(ggplot2) # Na potrzeby generowania wykresw
library(ggrepel) # Na potrzeby generowania wykresw
library(scales) # Na potrzeby generowania wykresw

# Wczytywanie danych; pierwsze cztery (zbdne) wiersze s pomijane.
wb_data <- read.csv(
  "data/world_bank_data.csv",
  stringsAsFactors = F,
  skip = 4
)

# Wizualne porwnanie wydatkw w latach 1990 i 2014.
# Rozpoczcie od przefiltrowania wierszy pod ktem analizowanego wskanika.
indicator <- "Government expenditure on education, total (% of GDP)"
expenditure_plot_data <- wb_data %>%
  filter(Indicator.Name == indicator)


# Tworzenie wykresu wydatkw w latach 1990 i 2014 za pomoc pakietu ggplot2.
# Wicej informacji znajdziesz w rozdziale 16.
expenditure_chart <- ggplot(data = expenditure_plot_data) +
  geom_text_repel(
    mapping = aes(x = X1990 / 100, y = X2014 / 100, label = Country.Code)
  ) +
  scale_x_continuous(labels = percent) +
  scale_y_continuous(labels = percent) +
  labs(
    title = "Wydatki rzdowe na edukacj w sumie (% PKB)",
    x = "Wydatki w 1990 r.", y = "Wydatki w 2014 r."
  )

# Zmiana ksztatu danych, aby utworzy now kolumn year.
long_year_data <- wb_data %>%
  gather(
    key = year,    # year bdzie now kolumn z kluczem.
    value = value, # value bdzie now kolumn z wartociami.
    X1960:X        # Zbierane s wszystkie kolumny od X1960 do X.
  )


# Filtrowanie wierszy pod ktem wskanika i kraju.
indicator <- "Government expenditure on education, total (% of GDP)"
spain_plot_data <- long_year_data %>%
  filter(
    Indicator.Name == indicator,
    Country.Code == "ESP" # Hiszpania
  ) %>%
  mutate(year = as.numeric(substr(year, 2, 5))) # Usuwanie "X" przed kadym rokiem.

# Wywietlanie wydatkw na edukacj w kolejnych latach.
chart_title <- paste("Wydatki rzdowe na edukacj w sumie (% PKB)", " - Hiszpania")
spain_chart <- ggplot(data = spain_plot_data) +
  geom_line(mapping = aes(x = year, y = value / 100)) +
  scale_y_continuous(labels = percent) +
  labs(title = chart_title, x = "Rok", y = "Wydatki (% PKB)")

# Zmiana ksztatu danych w celu utworzenia kolumny dla kadego wskanika.
wide_data <- long_year_data %>%
  select(-Indicator.Code) %>% # Naley pomin kolumn Indicator.Code.
  spread(
    key = Indicator.Name, # Nazwy nowych kolumn to wartoci z kolumny Indicator.Name.
    value = value # Zapenianie nowych kolumn wartociami z kolumny value.
  )

# Przygotowywanie danych i filtrowanie pod ktem roku.
x_var <- "Literacy rate, adult female (% of females ages 15 and above)"
y_var <- "Unemployment, female (% of female labor force) (modeled ILO estimate)"
lit_plot_data <- wide_data %>%
  mutate(
    lit_percent_2014 = wide_data[, x_var] / 100,
    employ_percent_2014 = wide_data[, y_var] / 100
  ) %>%
  filter(year == "X2014")

# Wywietlanie zalenoci midzy poziomem pimiennoci a stop bezrobocia.
lit_chart <- ggplot(data = lit_plot_data) +
  geom_point(mapping = aes(x = lit_percent_2014, y = employ_percent_2014)) +
  scale_x_continuous(labels = percent) +
  scale_y_continuous(labels = percent) +
  labs(
    x = "Wskanik pimiennoci wrd dorosych kobiet (% kobiet powyej 15 roku ycia)",
    y = "Bezrobocie wrd kobiet (% kobiet w wieku produkcyjnym)",
    title = "Zaleno midzy poziomem pimiennoci a stop bezrobocia wrd kobiet"
  )
