"""Czysty kod w Pythonie - Rozdział 7: Generatory, iteratory i programowanie asynchroniczne

> Iteracje idiomatyczne z wykorzystaniem modułu itertools

"""
import logging
from itertools import filterfalse, tee

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


class IteratorWrapper:
    def __init__(self, iterable):
        self.iterable = iter(iterable)

    def __next__(self):
        value = next(self.iterable)
        logger.debug(
            "%s Tworzenie następnej wartości: %r", self.__class__.__name__, value
        )
        return value

    def __iter__(self):
        return self


def partition(condition, iterable):
    """Zwraca 2 nowe obiekty iterowalne

    true_cond, false_cond = partition(condition, iterable)

    * in true_cond, warunek jest prawdziwy dla wszystkich elementów obiektu iterowalnego
    * in false_cond, warunek jest fałszywy dla wszystkich elementów obiektu iterowalnego
    """
    for_true, for_false = tee(iterable)
    return filter(condition, for_true), filterfalse(condition, for_false)


iterable = IteratorWrapper(
    {"name": f"element_{i}", "index": i} for i in range(20)
)


def is_even(record):
    return record["index"] % 2 == 0


def show(records):
    return ", ".join(e["name"] for e in records)


if __name__ == "__main__":
    even, odd = partition(is_even, iterable)

    logger.info(
        "\n\tRekordy parzyste: %s\n\t Rekordy nieparzyste: %s", show(even), show(odd)
    )
