"""Czysty kod w Pythonie - Rozdział 6: Deskryptory

> Metody interfejsu deskryptora: __set_name__

"""
from log import logger


class DescriptorWithName:
    """Ten deskryptor wymaga jawnego ustawienia nazwy."""

    def __init__(self, name):
        self.name = name

    def __get__(self, instance, value):
        if instance is None:
            return self
        logger.info("pobranie atrybutu %r z obiektu %r", self.name, instance)
        return instance.__dict__[self.name]

    def __set__(self, instance, value):
        instance.__dict__[self.name] = value


class ClientClass:
    """
    >>> client = ClientClass()
    >>> client.descriptor = "wartość"
    >>> client.descriptor
    'wartość'

    >>> ClientClass.descriptor_2.name
    "nazwa, która nie odpowiada atrybutowi"
    """

    descriptor = DescriptorWithName("descriptor")
    descriptor_2 = DescriptorWithName(
        "nazwa, która nie odpowiada atrybutowi"
    )


class DescriptorWithAutomaticName(DescriptorWithName):
    """Ten deskryptor potrafi wywnioskować nazwę atrybutu, jeśli nie zostanie ona podana jawnie.
    Obsługuje również jawne ustawianie różnych nazw.
    """

    def __init__(self, name: str = None) -> None:
        self.name = name

    def __set_name__(self, owner, name):
        self.name = self.name or name


class NewClientClass:
    """
    >>> NewClientClass.descriptor_with_default_name.name
    'descriptor_with_default_name'

    >>> NewClientClass.named_descriptor.name
    'named_descriptor'

    >>> NewClientClass.descriptor_named_differently.name
    'a_different_name'

    >>> client = NewClientClass()
    >>> client.descriptor_named_differently = "foo"
    >>> client.__dict__["a_different_name"]
    'foo'

    >>> client.descriptor_named_differently
    'foo'

    >>> client.a_different_name
    'foo'
    """

    descriptor_with_default_name = DescriptorWithAutomaticName()
    named_descriptor = DescriptorWithAutomaticName("named_descriptor")
    descriptor_named_differently = DescriptorWithAutomaticName(
        "a_different_name"
    )
