"""Czysty kod w Pythonie - Rozdział 6: Deskryptory

> Różne formy impelemtacji desckryptorów (``__dict__`` kontra ``weakref``)

  - Implementacja z weakref
"""

from weakref import WeakKeyDictionary


class DescriptorClass:
    def __init__(self, initial_value):
        self.value = initial_value
        self.mapping = WeakKeyDictionary()

    def __get__(self, instance, owner):
        if instance is None:
            return self
        return self.mapping.get(instance, self.value)

    def __set__(self, instance, value):
        self.mapping[instance] = value


class ClientClass:
    """
    >>> client1 = ClientClass()
    >>> client2 = ClientClass()

    >>> client1.descriptor = "nowa wartość"

    client1 musi mieć nową wartość, natomiast client2 nadal musi korzystać z wartości
    domyślnej:

    >>> client1.descriptor
    'nowa wartość'
    >>> client2.descriptor
    'wartość domyślna'

    Zmiana wartości dla obiektu client2 nie ma wpływu na obiekt client1

    >>> client2.descriptor = "wartość dla obiektu client2"
    >>> client2.descriptor
    'wartość dla obiektu client2'
    >>> client2.descriptor != client1.descriptor
    True
    """

    descriptor = DescriptorClass("wartość domyślna")
