"""Czysty kod w Pythonie - Rozdział 5: Dekoratory

> Dekoratory klas

Zmiana implementacji serializacji zdarzeń poprzez zastosowanie dekoratora klas.
"""
from dataclasses import dataclass
from datetime import datetime


def hide_field(field) -> str:
    return "**redacted**"


def format_time(field_timestamp: datetime) -> str:
    return field_timestamp.strftime("%Y-%m-%d %H:%M")


def show_original(event_field):
    return event_field


class EventSerializer:
    """Stosuje transformacje do obiektu Event na podstawie właściwości oraz
    definicji funkcji do zastosowania do każdego z pól.
    """

    def __init__(self, serialization_fields: dict) -> None:
        """Utworzone z wykorzystaniem mapowania pomiędzy polami, a funkcjami.

        Przykład::

        >>> serialization_fields = {
        ...    "username": str.upper,
        ...    "name": str.title,
        ... }

        Oznacza, że yen obiekt wywołany za pomocą następującego kodu::

        >>> from types import SimpleNamespace
        >>> event = SimpleNamespace(username="usr", name="name")
        >>> result = EventSerializer(serialization_fields).serialize(event)

        Zwróci słownik, w którym::

        >>> result == {
        ...     "username": event.username.upper(),
        ...     "name": event.name.title(),
        ... }
        True

        """
        self.serialization_fields = serialization_fields

    def serialize(self, event) -> dict:
        """Pobiera wszystkie atrybuty z ``event``, stosuje transformacje do
        każdego z atrybutów i umieszcza w słoniku, który jest zwracany.
        """
        return {
            field: transformation(getattr(event, field))
            for field, transformation in self.serialization_fields.items()
        }


class Serialization:
    """Dekorator klasy utworzony z wykorzystaniem funkcji transformacji do zastosowania
    do pól egzemplarza klasy.
    """

    def __init__(self, **transformations):
        """Słownik ``transformations`` zawiera definicje opisujące sposób
        mapowania atrybutów egzemplarza klasy podczas serializacji.
        """
        self.serializer = EventSerializer(transformations)

    def __call__(self, event_class):
        """Wywoływana podczas stosowania do ``event_class``, spowoduje zastąpienie
        metody ``serialize`` tej klasy nową wersją wykorzystującą 
        egzemplarz serializera.
        """

        def serialize_method(event_instance):
            return self.serializer.serialize(event_instance)

        event_class.serialize = serialize_method
        return event_class


@Serialization(
    username=str.lower,
    password=hide_field,
    ip=show_original,
    timestamp=format_time,
)
@dataclass
class LoginEvent:
    username: str
    password: str
    ip: str
    timestamp: datetime
