"""Czysty kod w Pythonie - Rozdział 3: Ogólne cechy dobrego kodu

> Obsługa błędów - Wyjątki
"""
import logging
import time

from base import Connector, Event

logger = logging.getLogger(__name__)


def connect_with_retry(
    connector: Connector, retry_n_times: int, retry_backoff: int = 5
):
    """Próbuje nawiązać połączenie z <connector> ponawiając próbę
    <retry_n_times> razy. Czeka <retry_backoff> sekund pomiędzy kolejnymi
Próbami nawiązania połączenia.

    Jeśli połączenie się powiedzie, zwraca obiekt połączenia.
    Jeśli nawiązanie połączenia nie jest możliwe po wyczerpaniu prób, 
Zgłasza wyjątek ``ConnectionError``.

    :param connector: Obiekt zawierający metodę ``.connect()``.
    :param retry_n_times int: Liczba prób wywołania metody
                               ``connector.connect()``.
    :param retry_backoff int: Przerwa pomiędzy kolejnymi próbami.
    """

    for _ in range(retry_n_times):
        try:
            return connector.connect()
        except ConnectionError as e:
            logger.info(
                "%s: próba nawiązania nowego połączenia za %is", e, retry_backoff
            )
            time.sleep(retry_backoff)
    exc = ConnectionError(f"Nie można nawiązać połączenia po podjęciu {retry_n_times} prób")
    logger.exception(exc)
    raise exc


class DataTransport:
    """Przykład obiektu, który oddziela obsługę wyjątków według 
       poziomów abstrakcji.
    """

    _RETRY_BACKOFF: int = 5
    _RETRY_TIMES: int = 3

    def __init__(self, connector: Connector) -> None:
        self._connector = connector
        self.connection = None

    def deliver_event(self, event: Event):
        self.connection = connect_with_retry(
            self._connector, self._RETRY_TIMES, self._RETRY_BACKOFF
        )
        self.send(event)

    def send(self, event: Event):
        try:
            return self.connection.send(event.decode())
        except ValueError as e:
            logger.error("%r zawiera nieprawidłowe dane: %s", event, e)
            raise
