"""Czysty kod w Pythonie - Rozdział 7: Generatory, iteratory i programowanie asynchroniczne

> Metody interfejsu generatorów.

"""
import time

from log import logger


class DBHandler:
    """Symulacja odczytu z bazy danych stronami."""

    def __init__(self, db):
        self.db = db
        self.is_closed = False

    def read_n_records(self, limit):
        return [(i, f"row {i}") for i in range(limit)]

    def close(self):
        logger.debug("zamknięcie połączenia z bazą danych %r", self.db)
        self.is_closed = True


def stream_db_records(db_handler):
    """Przykład użycia metody .close()

    >>> streamer = stream_db_records(DBHandler("testdb"))  # doctest: +ELLIPSIS
    >>> len(next(streamer))
    10

    >>> len(next(streamer))
    10
    """
    try:
        while True:
            yield db_handler.read_n_records(10)
            time.sleep(0.1)
    except GeneratorExit:
        db_handler.close()


class CustomException(Exception):
    """Wyjątek modelu dziedziny."""


def stream_data(db_handler):
    """Test metody ``.throw()``.

    >>> streamer = stream_data(DBHandler("testdb"))
    >>> len(next(streamer))
    10
    """
    while True:
        try:
            yield db_handler.read_n_records(10)
        except CustomException as e:
            logger.info("zarządzany błąd %r, kontynuacja", e)
        except Exception as e:
            logger.info("nieobsłużony błąd %r, zatrzymanie", e)
            db_handler.close()
            break
