"""Czysty kod w Pythonie - Rozdział 4

Zasada otwarty-zamknięty

Przykład naruszenia zasady otwarty-zamknięty.

Ten kod nie spełnia zasady otwarty-zamknięty i należy go zrefaktoryzować.
"""
from dataclasses import dataclass


@dataclass
class Event:
    raw_data: dict


class UnknownEvent(Event):
    """Typ zdarzenia, którego nie można zidentyfikować na podstawie jego danych."""


class LoginEvent(Event):
    """Zdarzenie reprezentujące użytkownika, który właśnie zalogował się w systemie."""


class LogoutEvent(Event):
    """Zdarzenie reprezentujące użytkownika, który właśnie wylogował się z systemu."""


class SystemMonitor:
    """Identyfikuje zdarzenia, które wystąpiły w systemie

    >>> l1 = SystemMonitor({"before": {"session": 0}, "after": {"session": 1}})
    >>> l1.identify_event().__class__.__name__
    'LoginEvent'

    >>> l2 = SystemMonitor({"before": {"session": 1}, "after": {"session": 0}})
    >>> l2.identify_event().__class__.__name__
    'LogoutEvent'

    >>> l3 = SystemMonitor({"before": {"session": 1}, "after": {"session": 1}})
    >>> l3.identify_event().__class__.__name__
    'UnknownEvent'

    """

    def __init__(self, event_data):
        self.event_data = event_data

    def identify_event(self):
        if (
            self.event_data["before"]["session"] == 0
            and self.event_data["after"]["session"] == 1
        ):
            return LoginEvent(self.event_data)
        elif (
            self.event_data["before"]["session"] == 1
            and self.event_data["after"]["session"] == 0
        ):
            return LogoutEvent(self.event_data)

        return UnknownEvent(self.event_data)
