# import pandas, matplotlib, and statsmodels
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import statsmodels.api as sm
pd.set_option('display.width', 80)
pd.set_option('display.max_columns', 12)
pd.set_option('display.max_rows', 200)
pd.options.display.float_format = '{:,.0f}'.format
landtemps = pd.read_csv("dane/landtemps2019avgs.csv")
covidtotals = pd.read_csv("dane/covidtotals.csv", parse_dates=["lastdate"])
covidtotals.set_index("iso_code", inplace=True)

# show some of the temperature rows 
landtemps[['station','country','latabs','elevation','avgtemp']].\
  sample(10, random_state=1)

# generate some descriptive statistics on the temperatures data
landtemps.describe()
landtemps.avgtemp.skew()
landtemps.avgtemp.kurtosis()

# plot temperature averages
plt.hist(landtemps.avgtemp)
plt.axvline(landtemps.avgtemp.mean(), color='red', linestyle='dashed', linewidth=1)
plt.title("Średnie temperatury w stacjach (stopnie Celsjusza)")
plt.xlabel("Średnia temperatura")
plt.ylabel("Liczba stacji")
plt.show()

# run a qq-plot to examine where the distribution deviates from a normal distribution
sm.qqplot(landtemps[['avgtemp']].sort_values(['avgtemp']), line='s')
plt.title("Wykres kwantyl-kwantyl dla średnich temperatur")
plt.xlabel("Kwantyle teoretyczne")
plt.ylabel("Kwantyle z próby")
plt.show()

# show skewness and kurtosis for total_cases_pm
covidtotals.total_cases_pm.skew()
covidtotals.total_cases_pm.kurtosis()

# do a stacked histogram
showregions = ['Oceania / Aus','East Asia','Southern Africa', 'Western Europe']
regionsLabels = ['Australia/Oceania', 'Azja Wschodnia', 'Afryka Południowa', 'Europa Zachodnia']

def getcases(regiondesc):
  return covidtotals.loc[covidtotals.region==regiondesc, 
  'total_cases_pm']

plt.hist([getcases(k) for k in showregions],\
  color=['blue','mediumslateblue','plum','mediumvioletred'],\
  label=regionsLabels,\
  stacked=True)
plt.title("Liczba przypadków koronawirusa na milion mieszkańców\nw wybranych regionach świata")
plt.xlabel("Liczba przypadków na milion mieszkańców")
plt.ylabel("Częstość występowania")
plt.xticks(np.arange(0, 22500, step=2500))
plt.legend()
plt.show()

# show multiple histograms on one figure
fig, axes = plt.subplots(2, 2)
fig.suptitle("Liczba przypadków koronawirusa na milion mieszkańców\nw różnych regionach świata")
axes = axes.ravel()

for j, ax in enumerate(axes):
  ax.hist(covidtotals.loc[covidtotals.region==showregions[j]].\
    total_cases_pm, bins=5)
  ax.set_title(regionsLabels[j], fontsize=10)
  for tick in ax.get_xticklabels():
    tick.set_rotation(45)

plt.tight_layout()
fig.subplots_adjust(top=0.88)
plt.show()

