﻿using System.ComponentModel;

namespace BeehiveManagementSystem;

class Queen : Bee, INotifyPropertyChanged
{
    public event PropertyChangedEventHandler? PropertyChanged;

    protected void OnPropertyChanged(string name)
    {
        PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(name));
    }

    public bool HiveIsRunning { get; private set; } = true;
    public bool OutOfHoney { get { return !HiveIsRunning; } }


    private IWorker[] workers = new IWorker[0];
    private decimal eggs = 0;
    private decimal unassignedWorkers = 3;

    public bool CanAssignWorkers { get { return unassignedWorkers >= 1; } }

    public string StatusReport { get; private set; } = "";

    public override decimal CostPerShift
    {
        get { return Constants.QUEEN_COST_PER_SHIFT; }
    }

    public Queen() : base("Królowa")
    {
        AssignBee("Zbieraczka nektaru");
        AssignBee("Producentka miodu");
        AssignBee("Opiekunka jaj");
    }

    public void AssignBee(string? job)
    {
        switch (job)
        {
            case "Zbieraczka nektaru":
                AddWorker(new NectarCollector());
                break;
            case "Producentka miodu":
                AddWorker(new HoneyManufacturer());
                break;
            case "Opiekunka jaj":
                AddWorker(new EggCare(this));
                break;
        }
        UpdateStatusReport(true);
    }

    private void AddWorker(IWorker worker)
    {
        if (unassignedWorkers >= 1)
        {
            unassignedWorkers--;
            Array.Resize(ref workers, workers.Length + 1);
            workers[workers.Length - 1] = worker;
        }
    }

    private void UpdateStatusReport(bool allWorkersDidTheirJobs)
    {
        StatusReport = $"Raport o stanie skarbca:\n{HoneyVault.StatusReport}\n" +
        $"\nLiczba jaj: {eggs:0.00}\nNieprzypisane robotnice: {unassignedWorkers:0.00}\n" +
        $"{WorkerStatus("Zbieraczka nektaru")}\n{WorkerStatus("Producentka miodu")}" +
        $"\n{WorkerStatus("Opiekunka jaj")}\nROBOTNICE W SUMIE: {workers.Length}";

        if (!allWorkersDidTheirJobs)
            StatusReport += "\nOSTRZEŻENIE: NIE WSZYSTKIE ROBOTNICE WYKONAŁY SWOJĄ PRACĘ";

        OnPropertyChanged("StatusReport");
        OnPropertyChanged("CanAssignWorkers");
        OnPropertyChanged("HiveIsRunning");
        OnPropertyChanged("OutOfHoney");
    }

    private string WorkerStatus(string job)
    {
        int count = 0;
        foreach (IWorker worker in workers)
            if (worker.Job == job) count++;
        return $"{job}: {count}";
    }

    public void ReportEggConversion(decimal eggsToConvert)
    {
        if (eggs >= eggsToConvert)
        {
            eggs -= eggsToConvert;
            unassignedWorkers += eggsToConvert;
        }
    }

    public override bool WorkTheNextShift()
    {
        eggs += Constants.EGGS_PER_SHIFT;
        bool allWorkersDidTheirJobs = true;
        foreach (Bee worker in workers)
        {
            if (!worker.WorkTheNextShift())
                allWorkersDidTheirJobs = false;
        }
        HoneyVault.ConsumeHoney(unassignedWorkers * Constants.HONEY_PER_UNASSIGNED_WORKER);
        UpdateStatusReport(allWorkersDidTheirJobs);

        HiveIsRunning = base.WorkTheNextShift();
        return HiveIsRunning;
    }
}