#include <exception>
#include <iostream>     // cout
#include <xercesc/dom/DOM.hpp>
#include <xercesc/framework/LocalFileFormatTarget.hpp>
#include <xercesc/sax/SAXException.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include "animal.hpp"
#include "xerces_strings.hpp"

using namespace std;
using namespace xercesc;

/*
 * definicja XercesInitializer jak na listingu 14-8
 */

// narzdzie RAII zwalniajce zasb przy wychodzeniu z zasigu
template<typename T>
class DOMPtr {
public:
    DOMPtr(T* t) : t_(t) { }
    ~DOMPtr( ) { t_->release( ); }
    T* operator->( ) const { return t_; }
private:
    // blokowanie kopiowania i przypisywania
    DOMPtr(const DOMPtr&);
    DOMPtr& operator=(const DOMPtr&);
    T* t_;
};

// zgasza uytkownikowi bedy napotkane przy przetwarzaniu
// dokumentu przy yuyciu DOMBuilder-a
class CircusErrorHandler : public DOMErrorHandler {
public:
    bool handleError(const DOMError& e)
    {
        std::cout << toNative(e.getMessage( )) << "\n";
        return false;
    }
};

// zwraca warto elementu "name", potomnego wzgldem
// przekazanego elementu "animal".
const XMLCh* getAnimalName(const DOMElement* animal)
{
    static XercesString name = fromNative("name");

    // przegldanie wzw potomnych elementu
    DOMNodeList* children = animal->getChildNodes( );
    for ( size_t i = 0,
                 len = children->getLength( ); 
          i < len; 
          ++i ) 
    {
        DOMNode* child = children->item(i);
        if ( child->getNodeType( ) == DOMNode::ELEMENT_NODE &&
             static_cast<DOMElement*>(child)->getTagName( ) == name )
        {
            // mamy element "name"
            return child->getTextContent( );
        }
    }
    return 0;
}

int main( )
{
    try {
        // Inicjalizacja wntrznoci Xerces i pozyskanie obiektu DOMImplementation;
        // specjalny argument sygnalizuje ch stosowania funkcji Load and Save (LS)
        XercesInitializer   init;
        DOMImplementation*  impl = 
            DOMImplementationRegistry::getDOMImplementation(
                fromNative("LS").c_str( )
            );
        if (impl == 0) {
            cout << "nie mona utworzy implementacji DOM\n";
            return EXIT_FAILURE;
        }

        // konstrukcja parsera DOMBuilder dla dokumentu cyrk.xml.
        DOMPtr<DOMBuilder>  parser = 
            static_cast<DOMImplementationLS*>(impl)->
                createDOMBuilder(DOMImplementationLS::MODE_SYNCHRONOUS, 0);

        // wczenie przestrzeni nazw (zbdne w naszym przykadzie)
        parser->setFeature(XMLUni::fgDOMNamespaces, true);

        // rejestracja obsugi bdw
        CircusErrorHandler  err;
        parser->setErrorHandler(&err);

        // przetwarzanie pliku cyrk.xml; zamiast 
        // nazwy pliku mona poda URL dokumentu
        DOMDocument* doc = 
            parser->parseURI("cyrk.xml");

        // szukanie sonia Horacego: zaczynamy od
        // pozyskania wskanika elementu "animalList"
        DOMElement*  animalList = doc->getDocumentElement( );
        if (animalList->getTagName( ) != fromNative("animalList")) {
            cout << "niepoprawny element gwny: " 
                 << toNative(animalList->getTagName( ))
                 << "\n";
            return EXIT_FAILURE;
        }

        // przegldanie eementw potomnych "animal" 
        // w poszukiwaniu elementu sonia Horacego
        DOMNodeList* animals = 
            animalList->getElementsByTagName(fromNative("animal").c_str( ));
        for ( size_t i = 0, 
                     len = animals->getLength( );
              i < len;
              ++i )
        {
            DOMElement* animal = 
              static_cast<DOMElement*>(animals->item(i));
            const XMLCh* name = getAnimalName(animal);
            if (name != 0 && name == fromNative("Horacy")) {
                // mamy Horacego -- usuwamy jego element z dokumentu
                animalList->removeChild(animal);
                animal->release( ); // optional.
                break;
            }
        }

        // konstrukcja obiektu DOMWriter celem zapisania dokumentu cyrk.xml
        DOMPtr<DOMWriter> writer = 
            static_cast<DOMImplementationLS*>(impl)->createDOMWriter( );
        writer->setErrorHandler(&err);

        // zapis nowej wersji cyrk.xml.
        LocalFileFormatTarget file("cyrk.xml");
        writer->writeNode(&file, *animalList);
    } catch (const SAXException& e) {
        cout << "bd xml: " << toNative(e.getMessage( )) << "\n";
        return EXIT_FAILURE;
    } catch (const DOMException& e) {
        cout << "bd xml: " << toNative(e.getMessage( )) << "\n";
        return EXIT_FAILURE;
    } catch (const exception& e) {
        cout << e.what( ) << "\n";
        return EXIT_FAILURE;
    }
}