#include <cstddef>      // size_t
#include <exception>
#include <iostream>     // cout
#include <xercesc/dom/DOM.hpp>
#include <xercesc/parsers/XercesDOMParser.hpp>
#include <xercesc/sax2/DefaultHandler.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <xalanc/DOMSupport/XalanDocumentPrefixResolver.hpp>
#include <xalanc/XalanTransformer/XercesDOMWrapperParsedSource.hpp>
#include <xalanc/XercesParserLiaison/XercesParserLiaison.hpp>
#include <xalanc/XercesParserLiaison/XercesDOMSupport.hpp>
#include <xalanc/XPath/XObject.hpp>
#include <xalanc/XPath/XPathEvaluator.hpp>
#include "animal.hpp"
#include "xerces_strings.hpp"

using namespace std;
using namespace xercesc;
using namespace xalanc;

// narzdzie RAII inicjalizujce parser i infrastruktur XPath
// oraz zwalniajcy zasoby przy wyjciu z zasigu
class XPathInitializer {
public:
    XPathInitializer( ) 
    { 
        XMLPlatformUtils::Initialize( );
        XPathEvaluator::initialize( );
    }
    ~XPathInitializer( ) 
    { 
        XPathEvaluator::terminate( );
        XMLPlatformUtils::Terminate( );
    }
private:
    // blokada kopiowania i przypisywania
    XPathInitializer(const XPathInitializer&);
    XPathInitializer& operator=(const XPathInitializer&);
};

// klasa obiektu odbierajcego powiadomienia o bdach
class CircusErrorHandler : public DefaultHandler {
public:
    void error(const SAXParseException& e)
    {
        throw runtime_error(toNative(e.getMessage( )));
    }
    void fatalError(const SAXParseException& e) { error(e); }
};

int main( )
{
    try {
        // inicjalizacja Xerces i XPath oraz konstrukcja parsera DOM
        XPathInitializer    init;
        XercesDOMParser     parser;
    
        // rejestracja obsugi bdw
        CircusErrorHandler error;
        parser.setErrorHandler(&error);

        // przetwarzanie dokumentu cyrk.xml
        parser.parse(fromNative("cyrk.xml").c_str( ));
        DOMDocument* doc = parser.getDocument( );
        DOMElement*  animalList = doc->getDocumentElement( );

        // pozyskanie obiektu XalanDocument na podstawie obiektu doc
        XercesDOMSupport              support;
        XercesParserLiaison           liaison(support);
        XercesDOMWrapperParsedSource  src(doc, liaison, support);
        XalanDocument*                xalanDoc = src.getDocument( );

        // obliczenie wyraenia XPath wyuskujcego list wzw
        // tekstowych zawierajcych imiona zwierzt
        XPathEvaluator                evaluator;
        XalanDocumentPrefixResolver   resolver(xalanDoc);
        XercesString                  xpath = 
            fromNative("animalList/animal/name/child::text( )");
        XObjectPtr                    result =
            evaluator.evaluate( 
                support,         // DOMSupport
                xalanDoc,        // wze kontekstu
                xpath.c_str( ),  // wyraenie XPath
                resolver );
        const NodeRefListBase&        nodeset = result->nodeset( );


        // przegldanie listy wynikowej, wypisanie imion poszczeglnych zwierzt
        for ( size_t i = 0,
                     len = nodeset.getLength( );
              i < len;
              ++i )
        {
            const XMLCh* name = 
              nodeset.item(i)->getNodeValue( ).c_str( );
            std::cout << toNative(name) << "\n";
        }
    } catch (const DOMException& e) {
        cout << "bd xml: " << toNative(e.getMessage( )) << "\n";
        return EXIT_FAILURE;
    } catch (const exception& e) {
        cout << e.what( ) << "\n";
        return EXIT_FAILURE;
    }
}