/**************************************************************************************************
*
* \file G34_Non_Owning_Type_Erasure_2.cpp
* \brief Wytyczna 34.: Pamiętaj o kosztach konfiguracji związanych z rodzajem opakowań używanych we wzorcu Ukrywanie typu
*
* Copyright (C) 2022 Klaus Iglberger - wszystkie prawa zastrzeżone
*
* Ten plik należy do materiałów uzupełniających do książki "Projektowanie oprogramowania w języku C++"
* wydanej przez wydawnictwo Helion.
*
**************************************************************************************************/


//---- <Circle.h> ---------------------------------------------------------------------------------

class Circle
{
 public:
   explicit Circle( double radius )
      : radius_( radius )
   {}

   double radius() const { return radius_; }
   /* Kilka innych funkcji pobierających i pomocniczych związanych z okręgami */

 private:
   double radius_;
   /* Kilka kolejnych danych składowych */
};


//---- <Square.h> ---------------------------------------------------------------------------------

class Square
{
 public:
   explicit Square( double side )
      : side_( side )
   {}

   double side() const { return side_; }
   /* Kilka innych funkcji pobierających i pomocniczych związanych z kwadratami */

 private:
   double side_;
   /* Kilka kolejnych danych składowych */
};


//---- <Shape.h> ----------------------------------------------------------------------------------

#include <array>
#include <cstddef>
#include <memory>
#include <utility>

namespace detail {

class ShapeConcept  // Wzorzec projektowy Polimorfizm zewnętrzny
{
 public:
   virtual ~ShapeConcept() = default;
   virtual void draw() const = 0;
   virtual std::unique_ptr<ShapeConcept> clone() const = 0;  // Wzorzec projektowy Prototyp
   virtual void clone( ShapeConcept* memory ) const = 0;  // Wzorzec projektowy Prototyp
};

template< typename ShapeT, typename DrawStrategy > class NonOwningShapeModel;

template< typename ShapeT
        , typename DrawStrategy >
class OwningShapeModel : public ShapeConcept
{
 public:
   explicit OwningShapeModel( ShapeT shape, DrawStrategy drawer )
      : shape_{ std::move(shape) }
      , drawer_{ std::move(drawer) }
   {}

   void draw() const override { drawer_(shape_); }

   std::unique_ptr<ShapeConcept> clone() const override  // Wzorzec projektowy Prototyp
   {
      return std::make_unique<OwningShapeModel>( *this );
   }

   void clone( ShapeConcept* memory ) const
   {
      using Model = NonOwningShapeModel<ShapeT const,DrawStrategy const>;

      std::construct_at( static_cast<Model*>(memory), shape_, drawer_ );

      // lub:
      // auto* ptr =
      //    const_cast<void*>(static_cast<void const volatile*>(memory));
      // ::new (ptr) Model( shape_, drawer_ );
   }

 private:
   ShapeT shape_;
   DrawStrategy drawer_;
};

template< typename ShapeT
        , typename DrawStrategy >
class NonOwningShapeModel : public ShapeConcept
{
 public:
   NonOwningShapeModel( ShapeT& shape, DrawStrategy& drawer )
      : shape_{ std::addressof(shape) }
      , drawer_{ std::addressof(drawer) }
   {}

   void draw() const override { (*drawer_)(*shape_); }

   std::unique_ptr<ShapeConcept> clone() const override
   {
      using Model = OwningShapeModel<ShapeT,DrawStrategy>;
      return std::make_unique<Model>( *shape_, *drawer_ );
   }

   void clone( ShapeConcept* memory ) const override
   {
      std::construct_at( static_cast<NonOwningShapeModel*>(memory), *this );

      // lub:
      // auto* ptr = const_cast<void*>(static_cast<void const volatile*>(memory));
      // ::new (ptr) NonOwningShapeModel( *this );
   }

 private:
   ShapeT* shape_{ nullptr };
   DrawStrategy* drawer_{ nullptr };
};

} // Koniec przestrzeni nazw detail


class Shape;


class ShapeConstRef
{
 public:
   // Typy 'ShapeT' oraz 'DrawStrategy' potencjalnie będą cv-kwalifikowane;
   // referencje l-wartości uniemożliwiają użycie referencji r-wartości
   template< typename ShapeT
           , typename DrawStrategy >
   ShapeConstRef( ShapeT& shape
                , DrawStrategy& drawer )
   {
      using Model =
         detail::NonOwningShapeModel<ShapeT const,DrawStrategy const>;
      static_assert( sizeof(Model) == MODEL_SIZE, "Wykryto nieprawidłowy rozmiar" );
      static_assert( alignof(Model) == alignof(void*), "Wykryto nieprawidłowe wyrównanie" );

      std::construct_at( static_cast<Model*>(pimpl()), shape, drawer );

      // lub:
      // auto* ptr =
      //    const_cast<void*>(static_cast<void const volatile*>(pimpl()));
      // ::new (ptr) Model( shape, drawer );
   }

   ShapeConstRef( Shape& other );
   ShapeConstRef( Shape const& other );

   ShapeConstRef( ShapeConstRef const& other )
   {
      other.pimpl()->clone( pimpl() );
   }

   ShapeConstRef& operator=( ShapeConstRef const& other )
   {
      // Idiom kopiuj i zamień
      ShapeConstRef copy( other );
      raw_.swap( copy.raw_ );
      return *this;
   }

   ~ShapeConstRef()
   {
      std::destroy_at( pimpl() );
      // lub: pimpl()->~ShapeConcept();
   }

   // Operacja przenoszenia nie jest jawnie deklarowana

 private:
   friend void draw( ShapeConstRef const& shape )
   {
      shape.pimpl()->draw();
   }

   detail::ShapeConcept* pimpl()  // Wzorzec projektowy Most
   {
      return reinterpret_cast<detail::ShapeConcept*>( raw_.data() );
   }

   detail::ShapeConcept const* pimpl() const
   {
      return reinterpret_cast<detail::ShapeConcept const*>( raw_.data() );
   }

   // Oczekiwana wielkość konkretyzacji modelu:
   // sizeof(ShapeT*) + sizeof(DrawStrategy*) + sizeof(vptr)
   static constexpr size_t MODEL_SIZE = 3U*sizeof(void*);

   alignas(void*) std::array<std::byte,MODEL_SIZE> raw_;

   friend class Shape;
};


class Shape
{
 public:
   template< typename ShapeT
           , typename DrawStrategy >
   Shape( ShapeT shape, DrawStrategy drawer )
   {
      using Model = detail::OwningShapeModel<ShapeT,DrawStrategy>;
      pimpl_ = std::make_unique<Model>( std::move(shape)
                                      , std::move(drawer) );
   }

   Shape( Shape const& other )
      : pimpl_( other.pimpl_->clone() )
   {}

   Shape( ShapeConstRef const& other )
      : pimpl_{ other.pimpl()->clone() }
   {}

   Shape& operator=( Shape const& other )
   {
      // Idiom kopiuj i zamień
      Shape copy( other );
      pimpl_.swap( copy.pimpl_ );
      return *this;
   }

   ~Shape() = default;
   Shape( Shape&& ) = default;
   Shape& operator=( Shape&& ) = default;

 private:
   friend void draw( Shape const& shape )
   {
      shape.pimpl_->draw();
   }

   std::unique_ptr<detail::ShapeConcept> pimpl_;  // Wzorzec projektowy Most

   friend class ShapeConstRef;
};


ShapeConstRef::ShapeConstRef( Shape& other )
{
   other.pimpl_->clone( pimpl() );
}

ShapeConstRef::ShapeConstRef( Shape const& other )
{
   other.pimpl_->clone( pimpl() );
}


//---- <Main.cpp> ---------------------------------------------------------------------------------

//#include <Circle.h>
//#include <Shape.h>
#include <cstdlib>

int main()
{
   // Tworzymy okrąg jako reprezentanta konkretnego typu figur
   Circle circle{ 3.14 };

   // Tworzymy strategię rysowania w formie wyrażenia lambda
   auto drawer = []( Circle const& c ){ /*...*/ };

   // Łączymy figurę i strategię rysowania w abstrakcji 'Shape'
   Shape shape1( circle, drawer );

   // Rysujemy figurę
   draw( shape1 );

   // Tworzymy referencję do figury
   // To działa, lecz referencja do figury zawiera wskaźnik na 
   // instancję 'shape1', a nie wskaźnik na 'circle'
   ShapeConstRef shaperef( shape1 );

   // Rysujemy z użyciem referencji shaperef, uzyskując takie same wyniki
   // To działa, jednak wymaga dwóch odwołań
   draw( shaperef );

   // Tworzymy głęboką kopię figury z użyciem referencji shaperef.
   // To NIE jest możliwe w przypadku użycia prostej, pozbawionej prawa własności 
   // implementacji wzorca. W razie zastosowania tej prostej implementacji
   // poniższy wiesz kodu tworzy kopię instancji 'shaperef'. Zmienna 'shape2'
   // sama stanowiłaby referencję, więc mielibyśmy już trzy poziomy odwołań… Ech!
   Shape shape2( shaperef );

   // Narysowanie kopii da takie same wyniki
   draw( shape2 );

   return EXIT_SUCCESS;
}

