/**************************************************************************************************
*
* \file G33_Small_Buffer_Optimization.cpp
* \brief Wytyczna 33.: Miej świadomość optymalizacyjnego potencjału wzorca projektowego Ukrywanie typu
*
* Copyright (C) 2022 Klaus Iglberger - wszystkie prawa zastrzeżone
*
* Ten plik należy do materiałów uzupełniających do książki "Projektowanie oprogramowania w języku C++"
* wydanej przez wydawnictwo Helion.
*
**************************************************************************************************/


//---- <Circle.h> ---------------------------------------------------------------------------------

class Circle
{
 public:
   explicit Circle( double radius )
      : radius_( radius )
   {}

   double radius() const { return radius_; }
   /* Kilka innych funkcji pobierających i pomocniczych związanych z okręgami */

 private:
   double radius_;
   /* Kilka kolejnych danych składowych */
};


//---- <Square.h> ---------------------------------------------------------------------------------

class Square
{
 public:
   explicit Square( double side )
      : side_( side )
   {}

   double side() const { return side_; }
   /* Kilka innych funkcji pobierających i pomocniczych związanych z kwadratami */

 private:
   double side_;
   /* Kilka kolejnych danych składowych */
};


//---- <Shape.h> ----------------------------------------------------------------------------------

#include <array>
#include <cstdlib>
#include <memory>
#include <utility>


template< size_t Capacity = 32U, size_t Alignment = alignof(void*) >
class Shape
{
 public:
   template< typename ShapeT, typename DrawStrategy >
   Shape( ShapeT shape, DrawStrategy drawer )
   {
      using Model = OwningModel<ShapeT,DrawStrategy>;

      static_assert( sizeof(Model) <= Capacity, "Użyty typ jest zbyt duży" );
      static_assert( alignof(Model) <= Alignment, "Użyty typ jest źle wyrównany" );

      std::construct_at( static_cast<Model*>(pimpl())
                       , std::move(shape), std::move(drawer) );
      // or:
      // auto* ptr =
      //    const_cast<void*>(static_cast<void const volatile*>(pimpl()));
      // ::new (ptr) Model( std::move(shape), std::move(drawer) );
   }

   Shape( Shape const& other )
   {
      other.pimpl()->clone( pimpl() );
   }

   Shape& operator=( Shape const& other )
   {
      // Idiom kopiuj i zamień
      Shape copy( other );
      buffer_.swap( copy.buffer_ );
      return *this;
   }

   Shape( Shape&& other ) noexcept
   {
      other.pimpl()->move( pimpl() );
   }

   Shape& operator=( Shape&& other ) noexcept
   {
      // Idiom kopiuj i zamień
      Shape copy( std::move(other) );
      buffer_.swap( copy.buffer_ );
      return *this;
   }

   ~Shape()
   {
      std::destroy_at( pimpl() );
      // lub: pimpl()->~Concept();
   }

 private:
   friend void draw( Shape const& shape )
   {
      shape.pimpl()->draw();
   }

   struct Concept  // Wzorzec projektowy Polimorfizm zewnętrzny
   {
      virtual ~Concept() = default;
      virtual void draw() const = 0;
      virtual void clone( Concept* memory ) const = 0;  // Wzorzec projektowy Prototyp
      virtual void move( Concept* memory ) = 0;
   };

   template< typename ShapeT, typename DrawStrategy >
   struct OwningModel : public Concept
   {
      OwningModel( ShapeT shape, DrawStrategy drawer )
         : shape_( std::move(shape) )
         , drawer_( std::move(drawer) )
      {}

      void draw() const override
      {
         drawer_( shape_ );
      }

      void clone( Concept* memory ) const override
      {
         std::construct_at( static_cast<OwningModel*>(memory), *this );

         // lub:
         // auto* ptr = const_cast<void*>(static_cast<void const volatile*>(memory));
         // ::new (ptr) OwningModel( *this );
      }

      void move( Concept* memory ) override
      {
         std::construct_at( static_cast<OwningModel*>(memory), std::move(*this) );

         // lub:
         // auto* ptr = const_cast<void*>(static_cast<void const volatile*>(memory));
         // ::new (ptr) OwningModel( std::move(*this) );
      }

      ShapeT shape_;
      DrawStrategy drawer_;
   };

   Concept* pimpl()  // Wzorzec projektowy Most
   {
      return reinterpret_cast<Concept*>( buffer_.data() );
   }

   Concept const* pimpl() const
   {
      return reinterpret_cast<Concept const*>( buffer_.data() );
   }

   alignas(Alignment) std::array<std::byte,Capacity> buffer_;
};


//---- <Main.cpp> ---------------------------------------------------------------------------------

//#include <Circle.h>
//#include <Square.h>
//#include <Shape.h>
#include <cstdlib>

int main()
{
   // Tworzymy okrąg jako jednego reprezentanta konkretnych typów figur
   Circle circle{ 3.14 };

   // Tworzymy strategię rysowania w formie wyrażenia lambda
   auto drawer = []( Circle const& c ){ /*...*/ };

   // Łączymy figurę oraz strategię rysowania w abstrakcji Shape.
   // Ten konstruktor skonkretyzuje typ 'deatail::OwningShapeModel' dla 
   // podanego typu 'Circle' oraz podanego typu wyrażenia lambda
   Shape shape1( circle, drawer );

   // Rysujemy figurę
   draw( shape1 );

   // Tworzymy kopię figury przy użyciu konstruktora kopiującego
   Shape shape2( shape1 );

   // Narysowanie tej kopii da takie same wyniki
   draw( shape2 );

   return EXIT_SUCCESS;
}

