/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference"
 * by Nicolai M. Josuttis, Addison-Wesley, 1999
 *
 * (C) Copyright Nicolai M. Josuttis 1999.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
/* klasa auto_ptr
 * - implementacja o zwiekszonej zgodnosci ze standardem
 */
namespace std {
    // typ pomocniczy w celu umozliwienia operacji kopiowania oraz przypisywania (teraz jest to obiekt globalny)
    template<class Y>
    struct auto_ptr_ref {
        Y* yp;
        auto_ptr_ref (Y* rhs)
         : yp(rhs) {
        }
    };

    template<class T>
    class auto_ptr {
      private:
        T* ap;    // wskazuje na rzeczywisty posiadany obiekt (jesli taki wystepuje)
      public:
        typedef T element_type;

        // konstruktor
        explicit auto_ptr (T* ptr = 0) throw()
         : ap(ptr) {
        }

        // konstruktory kopiujace (z niejawna konwersja)
        // - uwaga parametr nie jest typu const
        auto_ptr (auto_ptr& rhs) throw()
         : ap(rhs.release()) {
        }
        template<class Y>
        auto_ptr (auto_ptr<Y>& rhs) throw()
         : ap(rhs.release()) {
        }
        
        // przypsanie (z niejawna konwersja)
        // - uwaga parametr nie jest typu const
        auto_ptr& operator= (auto_ptr& rhs) throw() {
            reset(rhs.release());
            return *this;
        }
        template<class Y>
        auto_ptr& operator= (auto_ptr<Y>& rhs) throw() {
            reset(rhs.release());
            return *this;
        }
        
        // destruktor
        ~auto_ptr() throw() {
            delete ap;
        }

        // dostep do wartosc
        T* get() const throw() {
            return ap;
        }
        T& operator*() const throw() {
            return *ap;
        }
        T* operator->() const throw() {
            return ap;
        }

        // zwolnij posiadanie
        T* release() throw() {
            T* tmp(ap);
            ap = 0;
            return tmp;
        }

        // wyzeruj wartosc
        void reset (T* ptr=0) throw() {
            if (ap != ptr) {
                delete ap;
                ap = ptr;
            }
        }

        /* specjalne konwersje z typem pomocniczym w celu umozliwienia kopiowania oraz przypisywania
         */
        auto_ptr(auto_ptr_ref<T> rhs) throw()
         : ap(rhs.yp) {
        }
        auto_ptr& operator= (auto_ptr_ref<T> rhs) throw() {  // nowy obiekt
             reset(rhs.yp);
             return *this;
        }
        template<class Y>
        operator auto_ptr_ref<Y>() throw() {
            return auto_ptr_ref<Y>(release());
        }
        template<class Y>
        operator auto_ptr<Y>() throw() {
            return auto_ptr<Y>(release());
        }
    };
}
