/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference"
 * by Nicolai M. Josuttis, Addison-Wesley, 1999
 *
 * (C) Copyright Nicolai M. Josuttis 1999.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#include <cstdio>
#include <cstring>
#include <streambuf>

extern "C" {
    int read (int fd, char* buf, int num);
}

class inbuf : public std::streambuf {
  protected:
    /* bufora danych:
     * - co najwyzej, cztery znaki w obszarze przeznaczonym do ponownego wstawiania danych
     * - co najwyzej, szesc znakow w zwyklym buforze danych odczytanych
     */
    static const int bufferSize = 10;    // rozmiar bufora danych
    char buffer[bufferSize];             // bufor danych

  public:
    /* konstruktor
     * - zainicjalizuj pusty bufor danych
     * - brak obszaru sluzacego do ponownego wstawiania danych
     * => wymus wywolanie funkcji underflow()
     */
    inbuf() {
        setg (buffer+4,     // poczatek obszaru do ponownego wstawiania danych
              buffer+4,     // pozycja odczytu
              buffer+4);    // pozycja koncowa
    }

  protected:
    // wstaw do bufora nowe znaki
    virtual int_type underflow () {

        // czy pozycja jest przed koncem bufora?
        if (gptr() < egptr()) {
            return traits_type::to_int_type(*gptr());
        }

        /* przetworz obszar przeznaczony do wstawiania danych
         * - uzyj liczby odczytanych znakow
         * - jednak najwyzej cztery
         */
        int numPutback;
        numPutback = gptr() - eback();
        if (numPutback > 4) {
            numPutback = 4;
        }

        /* kopiuj do czterech znakow poprzednio wczytanych do bufora
         * uzywanego do ponownego umieszcznia danych (obszar pierwszych czterech znakow)
         */
        std::memmove (buffer+(4-numPutback), gptr()-numPutback,
                      numPutback);

        // odczytaj nowe znaki
        int num;
        num = read (0, buffer+4, bufferSize-4);
        if (num <= 0) {
            // BLAD lub EOF
            return EOF;
        }

        // wyzeruj wskazniki bufora
        setg (buffer+(4-numPutback),   // poczatek obszaru do ponownego wstawiania danych
              buffer+4,                // pozycja odczytu
              buffer+4+num);           // koniec bufora

        // zwroc kolejny znak
        return traits_type::to_int_type(*gptr());
    }
};
