// Ten program pokazuje zorientowane obiektowo podejście do
// listy dwukierunkowej. Lista jest delegowana do węzła.
// Węzeł jest abstrakcyjnym typem danych. Stosowane są trzy
// typy węzłów: początkowy, końcowy i wewnętrzny.
// Tylko węzły wewnętrzne przechowują dane.
//
// Klasa Data została utworzona w celu działania w charakterze
// obiektu przechowywanego na liście dwukierunkowej.
//
#include <iostream>

enum { kIsSmaller, kIsLarger, kIsSame };

// Klasa Data przeznaczona do użycia na liście.
// Każda klasa na tej liście musi obsługiwać dwie
// metody: show() wyświetlającą wartość oraz compare(), która
// zwraca relatywne położenie.
class Data
{
public:
    Data(int newVal):value(newVal) {}
    ~Data() {}
    int compare(const Data&);
    void show() { std::cout << value << "\n"; }
private:
    int value;
};

// Metoda decydująca, gdzie na liście zostanie
// umieszczony określony obiekt.
int Data::compare(const Data& otherData)
{
    if (value < otherData.value)
        return kIsSmaller;
    if (value > otherData.value)
        return kIsLarger;
    else
        return kIsSame;
}

// Wcześniejsze deklaracje klas.
class Node;
class HeadNode;
class TailNode;
class InternalNode;

// To jest abstrakcyjny typ danych przedstawiający obiekt węzła na liście.
// Każda klasa potomna musi nadpisywać metody insert() i show().
class Node
{
public:
    Node() {}
    virtual ~Node() {}
    virtual Node* insert(Data* data) = 0;
    virtual void show() = 0;
private:
};

// To jest węzeł przechowujący rzeczywisty obiekt.
// W tym programie to jest obiekt typu Data.
// Bardziej ogólne podejście poznasz, gdy zajmiemy
// się poznaniem szablonów.
class InternalNode : public Node
{
public:
    InternalNode(Data* data, Node* next);
    virtual ~InternalNode() { delete next; delete data; }
    virtual Node* insert(Data* data);
    virtual void show()
        { data->show(); next->show(); } // Delegacja!

private:
    Data* data;  // Właściwe dane.
    Node* next;  // Ten wskaźnik prowadzi do następnego węzła na liście.
};

// Zadanie tego konstruktora sprowadza się tylko do inicjalizacji.
InternalNode::InternalNode(Data* newData, Node* newNext):
data(newData), next(newNext)
{
}

// To jest istota listy.
// Metoda przeznaczona do przechowywania nowego obiektu na liście.
// Obiekt zostaje przekazany do węzła, który ustala miejsce jego
// umieszczenia, a następnie faktycznie wstawia obiekt na liście.
Node* InternalNode::insert(Data* otherData)
{
    // Czy nowy obiekt jest mniejszy, czy większy od bieżącego?
    int result = data->compare(*otherData);

    switch(result)
    {
    // Jeżeli są takie same, wtedy zgodnie z konwencją nowy będzie pierwszym.
    case kIsSame:      // Przejście dalej.
    case kIsLarger:    // Nowe dane zostaną umieszczone przed bieżącymi.
        {
            InternalNode* dataNode =
                new InternalNode(otherData, this);
            return dataNode;
        }

        // Nowy jest większy, więc zostaje przekazany do następnego
        // węzła, gdzie powinien być obsłużony.
    case kIsSmaller:
        next = next->insert(otherData);
        return this;
    }
    return this;  // Uspokojenie kompilatora.
}

// Ostatni węzeł na liście.
class TailNode : public Node
{
public:
    TailNode() {}
    virtual ~TailNode() {}
    virtual Node* insert(Data* data);
    virtual void show() {}
private:
};

// Jeżeli dane są przeznaczone dla tego węzła, muszą być wstawione przed nim,
// ponieważ nic nie może się znajdować po węźle końcowym.
Node* TailNode::insert(Data* data)
{
    InternalNode* dataNode = new InternalNode(data, this);
    return dataNode;
}

// Węzeł początkowy nie przechowuje żadnych danych, ale
// wskazuje początek listy.
class HeadNode : public Node
{
public:
    HeadNode();
    virtual ~HeadNode() { delete next; }
    virtual Node* insert(Data* data);
    virtual void show() { next->show(); }
private:
    Node* next;
};

// Pierwszy węzeł na liście powoduje utworzenie węzła końcowego.
HeadNode::HeadNode()
{
    next = new TailNode;
}

// Ponieważ nic nie może się znajdować przed węzłem początkowym,
// dane przekazujemy po prostu do następnego węzła.
Node* HeadNode::insert(Data* data)
{
    next = next->insert(data);
    return this;
}

// Ta klasa otrzymuje uznanie, choć nie wykonuje żadnej pracy.
class LinkedList
{
public:
    LinkedList();
    ~LinkedList() { delete head; }
    void insert(Data* data);
    void showAll() { head->show(); }
private:
    HeadNode* head;
};

// W chwili powstania następuje utworzenie węzła początkowego,
// który z kolei tworzy węzeł końcowy.
// Dlatego też pusta lista prowadzi do węzła początkowego, który
// z kolei prowadzi do węzła końcowego, a między nimi nie ma niczego.
LinkedList::LinkedList()
{
    head = new HeadNode;
}

// Delegacja do węzła początkowego.
void LinkedList::insert(Data* pData)
{
    head->insert(pData);
}

// Zebranie wszystkich klas w celu ich przetestowania.
int main()
{
    Data* pData;
    int val;
    LinkedList ll;

    // Prosimy użytkownika o wprowadzenie pewnych wartości,
    // które następnie zostaną umieszczone na liście.
    while (true)
    {
        std::cout << "Podaj wartość (0 oznacza zakończenie programu): ";
        std::cin >> val;
        if (!val)
            break;
        pData = new Data(val);
        ll.insert(pData);
   }

   // Iteracja przez listę i wyświetlenie jej danych.
   ll.showAll();
   return 0;  // Obiekt ll zostaje usunięty z pamięci i zniszczony!
}
