/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference, 2nd Edition"
 * by Nicolai M. Josuttis, Addison-Wesley, 2012
 *
 * (C) Copyright Nicolai M. Josuttis 2012.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#include <iostream>
#include <string>
#include <set>
#include <deque>
#include <algorithm>
#include <memory>

class Item {
  private:
    std::string name;
    float  price;
  public:
    Item (const std::string& n, float p = 0) : name(n), price(p) {
    }
    std::string getName () const {
        return name;
    }
    void setName (const std::string& n) {
        name = n;
    }
    float getPrice () const {
        return price;
    }
    float setPrice (float p) {
        price = p;
    }
};

template <typename Coll>
void printItems (const std::string& msg, const Coll& coll)
{
    std::cout << msg << std::endl;
    for (const auto& elem : coll) {
        std::cout << ' ' << elem->getName() << ": "
                         << elem->getPrice() << std::endl;
    }
}

int main()
{
    using namespace std;

    // dwie różne kolekcje współdzielące Items
    typedef shared_ptr<Item> ItemPtr;
    set<ItemPtr> allItems;
    deque<ItemPtr> bestsellers;

    // wstawianie obiektów do kolekcji
    // - bestsellery znajdują się w obu kolekcjach
    bestsellers = { ItemPtr(new Item("Sen marynarza",20.10)),
                    ItemPtr(new Item("Morderstwo o poranku",14.99)),
                    ItemPtr(new Item("Moje rozkosze",9.88)) };
    allItems = { ItemPtr(new Item("Woda",0.44)),
                 ItemPtr(new Item("Pizza",2.22)) };
    allItems.insert(bestsellers.begin(),bestsellers.end());

    // wyświetl zawartość obu kolekcji
    printItems ("bestsellery:", bestsellers);
    printItems ("wszystkie:", allItems);
    cout << endl;

    // podwojenie ceny bestsellerów
    for_each (bestsellers.begin(), bestsellers.end(),
              [] (shared_ptr<Item>& elem) {
                  elem->setPrice(elem->getPrice() * 2);
              });
    
    // zastąp drugi bestseller pierwszym elementem o nazwie "Pizza"
    bestsellers[1] = *(find_if(allItems.begin(),allItems.end(),
                               [] (shared_ptr<Item> elem) {
                                   return elem->getName() == "Pizza";
                               }));

    // ustaw cenę pierwszego bestsellera
    bestsellers[0]->setPrice(44.77);

    // wyświetl zawartość obu kolekcji
    printItems ("bestsellery:", bestsellers);
    printItems ("wszystkie:", allItems);
}

