/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference, 2nd Edition"
 * by Nicolai M. Josuttis, Addison-Wesley, 2012
 *
 * (C) Copyright Nicolai M. Josuttis 2012.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#ifndef ICSTRING_HPP
#define ICSTRING_HPP

#include <string>
#include <iostream>
#include <cctype>

// zastąp funkcje zdefiniowane w standardowej klasie char_traits<char>
// tak aby w przypadku łańcuchów znakowych wielkosc liter nie była rozróżniana
struct ignorecase_traits : public std::char_traits<char> {
    // sprawdź, czy łańcuchy c1 i c2 są jednakowe
    static bool eq(const char& c1, const char& c2) {
        return std::toupper(c1)==std::toupper(c2);
    }
    // sprawdź, czy łańcuch c1 jest “mniejszy” od łańcucha c2
    static bool lt(const char& c1, const char& c2) {
        return std::toupper(c1)<std::toupper(c2);
    }
    // porównaj pierwsze n znaków łańcuchów s1 oraz s2
    static int compare(const char* s1, const char* s2,
                       std::size_t n) {
        for (std::size_t i=0; i<n; ++i) {
            if (!eq(s1[i],s2[i])) {
                return lt(s1[i],s2[i])?-1:1;
            }
        }
        return 0;
    }
    // poszukaj znaków c w łańcuchu s
    static const char* find(const char* s, std::size_t n,
                            const char& c) {
        for (std::size_t i=0; i<n; ++i) {
            if (eq(s[i],c)) {
                return &(s[i]);
            }
        }
        return 0;
    }
};

// zdefiniuj specjalny typ dla tego rodzaju łańcuchów znakowych
typedef std::basic_string<char,ignorecase_traits> icstring;

// zdefiniuj operator zapisu do wyjścia
// ponieważ klasa cech typowych jest inna niż ta określona dla std::ostream
inline
std::ostream& operator << (std::ostream& strm, const icstring& s)
{
    // przekonwertuj łańcuch icstring do postaci zwykłego łańcucha
    return strm << std::string(s.data(),s.length());
}

#endif    // ICSTRING_HPP
