/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference, 2nd Edition"
 * by Nicolai M. Josuttis, Addison-Wesley, 2012
 *
 * (C) Copyright Nicolai M. Josuttis 2012.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#include <cstdio>
#include <cstring>
#include <streambuf>

// funkcja read():
#ifdef _MSC_VER
# include <io.h>
#else
# include <unistd.h>
#endif

class inbuf : public std::streambuf {

  protected:
    // bufor danych:
    // - co najwyżej, cztery znaki w obszarze przeznaczonym do ponownego wstawiania danych
    // - co najwyżej, szesc znakow w zwykłym buforze danych odczytanych
    static const int bufferSize = 10;    // rozmiar bufora danych
    char buffer[bufferSize];             // bufor danych

  public:
    // konstruktor
    // - zainicjalizuj pusty bufor danych
    // - brak obszaru slużącego do ponownego wstawiania danych
    // => wymuś wywołanie funkcji underflow()
    inbuf() {
        setg (buffer+4,     // początek obszaru do ponownego wstawiania danych
              buffer+4,     // pozycja odczytu
              buffer+4);    // pozycja końcowa
    }

  protected:
    // wstaw do bufora nowe znaki
    virtual int_type underflow () {

        // czy pozycja jest przed końcem bufora?
        if (gptr() < egptr()) {
            return traits_type::to_int_type(*gptr());
        }

        // przetwórz obszar przeznaczony do wstawiania danych
        // - użyj liczby odczytanych znaków
        // - ale najwyżej cztery
        int numPutback;
        numPutback = gptr() - eback();
        if (numPutback > 4) {
            numPutback = 4;
        }

        // kopiuj do czterech znaków poprzednio wczytanych do bufora
        // używanego do ponownego umieszcznia danych (obszar pierwszych czterech znaków)
        std::memmove (buffer+(4-numPutback), gptr()-numPutback,
                      numPutback);

        // odczytaj nowe znaki
        int num;
        num = read (0, buffer+4, bufferSize-4);
        if (num <= 0) {
            // BŁĄD lub EOF
            return EOF;
        }
        // wyzeruj wskaźniki bufora
        setg (buffer+(4-numPutback),   // początek obszaru do ponownego wstawiania danych
              buffer+4,                // pozycja odczytu
              buffer+4+num);           // koniec bufora

        // zwróć kolejny znak
        return traits_type::to_int_type(*gptr());
    }
};
