/* The following code example is taken from the book
 * "The C++ Standard Library - A Tutorial and Reference, 2nd Edition"
 * by Nicolai M. Josuttis, Addison-Wesley, 2012
 *
 * (C) Copyright Nicolai M. Josuttis 2012.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 */
#include <iostream>
#include <iomanip>
#include <map>
#include <string>
#include <algorithm>
#include <cctype>
using namespace std;

// obiekt funkcyjny do porównywania łańcuchów
// - umożliwia ustawianie kryterium porównania podczas wykonywania
// - umożliwia porównywanie bez rozróżniania wielkości liter
class RuntimeStringCmp {
  public:
    // stałe dla kryterium porównania
    enum cmp_mode {normal, nocase};
  private:
    // rzeczywisty tryb porównywania
    const cmp_mode mode;

    // funkcja pomocnicza do porównywania bez rozróżniania wielkości liter
    static bool nocase_compare (char c1, char c2) {
        return toupper(c1) < toupper(c2);
    }
  public:  
    // konstruktor: inicjalizuje kryterium porównania
    RuntimeStringCmp (cmp_mode m=normal) : mode(m) {
    }

    // operacja porównania
    bool operator() (const string& s1, const string& s2) const {
        if (mode == normal) {
            return s1<s2;
        }
        else {
            return lexicographical_compare (s1.begin(), s1.end(),
                                            s2.begin(), s2.end(),
                                            nocase_compare);
        }
    }
};

// typ kontenera:
// - mapa zawierająca
//       - klucze typu string
//       - wartości typu string
//       - specjalny typ obiektu realizującego porównania
typedef map<string,string,RuntimeStringCmp> StringStringMap;

// funkcja wypełniająca i wyświetlająca zawartość takich kontenerów
void fillAndPrint(StringStringMap& coll);

int main()
{
    // utwórz kontener o domyślnym kryterium porównania
    StringStringMap coll1;
    fillAndPrint(coll1);

    // utwórz obiekt realizujący porównania bez rozróżniania wielkości liter
    RuntimeStringCmp ignorecase(RuntimeStringCmp::nocase);

    // utwórz kontener o kryterium porównania nierozróżniającym wielkości liter
    StringStringMap coll2(ignorecase);
    fillAndPrint(coll2);
}

void fillAndPrint(StringStringMap& coll)
{
    // wstaw elementy w przypadkowej kolejności
    coll["Deutschland"] = "Niemcy";
    coll["deutsch"] = "niemiecki";
    coll["Haken"] = "haczyk";
    coll["arbeiten"] = "pracowac";
    coll["Hund"] = "pies";
    coll["gehen"] = "isc";
    coll["Unternehmen"] = "przedsiebiorstwo";
    coll["unternehmen"] = "podjac sie";
    coll["gehen"] = "isc";
    coll["Bestatter"] = "grabarz";

    // wypisz elementy
    cout.setf(ios::left, ios::adjustfield);
    for (const auto& elem : coll) {
        cout << setw(15) << elem.first << " "
             << elem.second << endl;
    }
    cout << endl;
}

