#include <stdlib.h>
#include <openssl/bn.h>

#define NUMBER_ITERS    5
#define NUMBER_PRIMES   100

static unsigned long primes[NUMBER_PRIMES] = {
  2,   3,   5,   7,   11,  13,  17,  19,  23,  29,  31,  37, 41,  43,  47,  53,
  59,  61,  67,  71,  73,  79,  83,  89,  97,  101, 103, 107, 109, 113, 127, 131,
  137, 139, 149, 151, 157, 163, 167, 173, 179, 181, 191, 193, 197, 199, 211, 223,
  227, 229, 233, 239, 241, 251, 257, 263, 269, 271, 277, 281, 283, 293, 307, 311,
  313, 317, 331, 337, 347, 349, 353, 359, 367, 373, 379, 383, 389, 397, 401, 409,
  419, 421, 431, 433, 439, 443, 449, 457, 461, 463, 467, 479, 487, 491, 499, 503,
  509, 521, 523, 541
};

static int is_obviously_not_prime(BIGNUM *p);
static int passes_rabin_miller_once(BIGNUM *p);
static unsigned int calc_b_and_m(BIGNUM *p, BIGNUM *m);

int spc_is_probably_prime(BIGNUM *p) {
  int i;

  if (is_obviously_not_prime(p)) return 0;
  for (i = 0;  i < NUMBER_ITERS;  i++)
    if (!passes_rabin_miller_once(p))
      return 0;
  return 1;
}

BIGNUM *spc_generate_prime(int nbits) {
  BIGNUM        *p = BN_new();
  unsigned char binary_rep[nbits / 8];

  /* W poniszym kodzie zakadamy, e zawsze bdziemy generowali liczby pierwsze
   * reprezentowane za pomoc liczby bitw bdcej wielokrotnoci omiu!
   */
  if (nbits % 8 || !p) abort();

  for (;;) {
    spc_rand(binary_rep, nbits / 8);

    /* Ustawia dwa najbardziej znaczce bity oraz jeden najmniej znaczcy bit. */
    binary_rep[0] |= 0xc0;
    binary_rep[nbits / 8 - 1] |= 1;

    /* Przeksztaca dan liczb do jej reprezentacji w postaci obiektu typu BIGNUM. */
    if (!BN_bin2bn(binary_rep, nbits / 8, p)) abort();

    /* Przed wywoaniem funkcji spc_is_probably_prime(p) planujemy testy zgodnoci
     * z wymaganiami algorytmu Diffiego-Hellmana odnonie wykorzystywanej liczby
     * pierwszej.
     */
    if (spc_is_probably_prime(p)) return p;
  }
}

/* Prbujemy najpierw dzielenia przez wszystkie nasze mae liczby pierwsze. Oznacza
 * to, e dla kadej z tych liczb pierwszych sprawdzamy, czy parzycie dzieli nasz
 * warto p  wwczas zwracamy 0. Nietrudno zauway, e ponisze rozwizanie nie
 * zadziaa prawidowo, jeli sprbujemy w ten sposb sprawdzi liczb pierwsz,
 * ktra znajduje si na naszej licie!
 */
static int is_obviously_not_prime(BIGNUM *p) {
  int i;

  for (i = 0;  i < NUMBER_PRIMES;  i++)
    if (!BN_mod_word(p, primes[i])) return 1;
  return 0;
}

static int passes_rabin_miller_once(BIGNUM *p) {
  BIGNUM       a, m, z, tmp;
  BN_CTX       *ctx;
  unsigned int b, i;

  /* Inicjalizuje obiekty a, m, z oraz tmp. */
  BN_init(&a);
  BN_init(&m);
  BN_init(&z);
  BN_init(&tmp);

  ctx = BN_CTX_new();
  b = calc_b_and_m(p, &m);

  /* a jest liczb losow mniejsz od p: */
  if (!BN_rand_range(&a, p)) abort();

  /* z = a^m mod p. */
  if (!BN_mod_exp(&z, &a, &m, p, ctx)) abort();

  /* Jeli na pocztku z = 1, pomi dalsze kroki. */
  if (BN_is_one(&z)) return 1;

  for (i = 0;  i < b;  i++) {
    if (BN_is_one(&z)) return 0;

    /* Jeli z = p - 1, pomi dalsze kroki! */
    BN_copy(&tmp, &z);
    if (!BN_add_word(&tmp, 1)) abort();
    if (!BN_cmp(&tmp, p)) return 1;

    /* z = z^2 mod p */
    BN_mod_sqr(&tmp, &z, p, ctx);
    BN_copy(&z, &tmp);
  }

  /* Jeli z = p - 1, pomi dalsze kroki! */
  BN_copy(&tmp, &z);
  if (!BN_add_word(&tmp, 1)) abort();
  if (!BN_cmp(&tmp, p)) return 1;

  /* Liczba zoona! */
  return 0;
}

/* b = liczba moliwych operacji cakowitego dzielenia liczby 2 przez p - 1. Wanie
 * ta warto jest zwracana przez ponisz funkcj. m = (p-1)/(2^b).
 */
static unsigned int calc_b_and_m(BIGNUM *p, BIGNUM *x) {
  unsigned int b;

  if (!BN_copy(x, p)) abort();
  if (!BN_sub_word(x, 1))  abort();

  for (b = 0;  !BN_is_odd(x);  b++)
    BN_div_word(x, 2);
  return b;
}

