<?php
App::import('Core', 'DboMysql');
class DboMysqlTransaction extends DboMysql {
	/**
	 * Backup of autoCommit status
	 *
	 * @var boolean
	 */
	protected $backAutoCommit;

	/**
	 * Error code for lock wait timeout exceeded messages
	 *
	 * @var int
	 */
	protected $lockTimeoutErrorCode = 1205;

	/**
	 * Constructor
	 */
	public function __construct($config = null, $autoConnect = true) {
		$this->_baseConfig = Set::merge(array(
			'lock' => array(
				'log' => LOGS . 'locks.log',
				'recover' => true,
				'retries' => 1
			),
			'autoCommit' => null
		), $this->_baseConfig);

		$this->_commands = array_merge(array(
			'lock' => 'LOCK TABLES {$table} {$operation}',
			'unlock' => 'UNLOCK TABLES',
			'setAutoCommit' => 'SET @@autoCommit={$autoCommit}'
		), $this->_commands);

		parent::__construct($config, $autoConnect);

		if (
			!is_null($this->config['autoCommit']) &&
			!$this->setAutoCommit($this->config['autoCommit'])
		) {
			trigger_error('Could not set autoCommit', E_USER_WARNING);
		}
	}

	/**
	 * Lock a table
	 *
	 * @param AppModel Model
	 * @param array $options Options (table, alias, operation)
	 */
	public function lock($model = null, $options = array()) {
		if (!is_object($model) && empty($options)) {
			$options = $model;
			$model = null;
		}

		if (empty($options) && !isset($model)) {
			trigger_error('Nothing to lock', E_USER_WARNING);
			return false;
		} elseif (!is_array($options)) {
			$options = array('table' => $options);
		} elseif (Set::numeric(array_keys($options))) {
			if (count($options) > 1) {
				$options = array('table' => $options[0], 'operation' => $options[1]);
			} else {
				if (!empty($options[0]) && is_array($options[0])) {
					$options = $options[0];
				} else {
					$options = array('table' => $options[0]);
				}
			}
		}

		if (empty($options['table']) && isset($model)) {
			$options = array_merge(array(
				'table' => $model->table,
				'alias' => $model->alias
			), $options);

			if (!empty($options['operation']) && $options['operation'] == 'read') {
				unset($options['alias']);
			}
		}

		$options = array_merge(array('alias'=>null, 'operation'=>'read', 'local'=>false, 'low'=>false), $options);
		if (!in_array(strtolower($options['operation']), array('read', 'write'))) {
			trigger_error(sprintf('Invalid operation %s for locking', $options['operation']), E_USER_WARNING);
			return false;
		}

		$table = $this->fullTableName($options['table']);
		if (!empty($options['alias'])) {
			$table .= ' AS ' . $this->name($options['alias']);
		}
		$operation = strtoupper($options['operation']);
		if ($options['operation'] == 'read' && $options['local']) {
			$operation .= ' LOCAL';
		} elseif ($options['operation'] == 'write' && $options['low']) {
			$operation = 'LOW_PRIORITY ' . $operation;
		}

		$sql = strtr($this->_commands['lock'], array(
			'{$table}' => $table,
			'{$operation}' => $operation
		));
		return ($this->query($sql) !== false);
	}

	/**
	 * Unlock tables
	 *
	 * @param AppModel Model
	 * @param array $options Options (not utilized)
	 */
	public function unlock($model = null, $options = array()) {
		return ($this->query($this->_commands['unlock']) !== false);
	}

	/**
	 * Get autoCommit status
	 *
	 * @param AppModel Model
	 * @return boolean true if autoCommit is on, false otherwise
	 */
	public function getAutoCommit($model = null) {
		if (is_null($this->config['autoCommit'])) {
			if (!$this->isConnected() && !$this->connect()) {
				trigger_error('Could not connect to database', E_USER_WARNING);
				return false;
			}

			$result = $this->query('SELECT @@autocommit AS ' . $this->name('autocommit'));
			if (empty($result)) {
				trigger_error('Could not fetch autoCommit status from database', E_USER_WARNING);
				return false;
			}
			$this->config['autoCommit'] = !empty($result[0][0]['autocommit']);
		}
		return $this->config['autoCommit'];
	}

	/**
	 * Set autoCommit status
	 *
	 * @param AppModel Model
	 * @param boolean $autoCommit true if autoCommit is on, false otherwise
	 * @return boolean Success
	 */
	public function setAutoCommit($model, $autoCommit = null) {
		if (!$this->isConnected() && !$this->connect()) {
			trigger_error('Could not connect to database', E_USER_WARNING);
			return false;
		}

		if (is_bool($model)) {
			$autoCommit = $model;
			$model = null;
		} elseif (is_array($autoCommit)) {
			list($autoCommit) = $autoCommit;
		}

		$this->config['autoCommit'] = !empty($autoCommit);
		$sql = strtr($this->_commands['setAutoCommit'], array(
			'{$autoCommit}' => ($this->config['autoCommit'] ? '1' : '0')
		));
		return ($this->query($sql) !== false);
	}

	/**
	 * Begin transaction
	 *
	 * @param AppModel model
	 * @return boolean Success
	 */
	public function begin($model) {
		$this->_startTransaction();
		return parent::begin($model);
	}

	/**
	 * Commit transaction
	 *
	 * @param AppModel model
	 * @return boolean Success
	 */
	public function commit($model) {
		$result = parent::commit($model);
		$this->_endTransaction();
		return $result;
	}

	/**
	 * Rollback transaction
	 *
	 * @param AppModel model
	 * @return boolean Success
	 */
	public function rollback($model) {
		$result = parent::rollback($model);
		$this->_endTransaction();
		return $result;
	}

	/**
	 * Setup prior to starting a transaction
	 */
	protected function _startTransaction() {
		if ($this->getAutoCommit()) {
			$this->backAutoCommit = $this->getAutoCommit();
			$this->setAutoCommit(false);
		}
	}

	/**
	 * Cleanup after finishing a transaction
	 */
	protected function _endTransaction() {
		if (isset($this->backAutoCommit)) {
			$this->setAutoCommit($this->backAutoCommit);
			$this->backAutoCommit = null;
		}
	}

	/**
	 * DataSource Query abstraction
	 *
	 * @return resource Result resource identifier
	 */
	public function query() {
		$args = func_get_args();
		if (!empty($args) && count($args) > 2 && in_array($args[0], array_keys($this->_commands))) {
			list($command, $params, $model) = $args;
			if ($this->isInterfaceSupported($command)) {
				return $this->{$command}($model, $params);
			}
		}
		return call_user_func_array(array('parent', 'query'), $args);
	}

	/**
	 * Executes given SQL statement.
	 *
	 * @param string $sql SQL statement
	 * @param int $retry Which retry is this
	 * @return resource Result resource identifier
	 */
	public function _execute($sql, $retry = 0) {
		$result = parent::_execute($sql);
		$error = $this->lastError();
		if (
			!empty($error) &&
			$this->config['lock']['recover'] &&
			preg_match('/^\b' . preg_quote($this->lockTimeoutErrorCode) . '\b/', $error)
		) {
			if ($retry == 0) {
				$message = 'Got lock on query [' . $sql . ']';
				$queries = array_reverse(Set::extract($this->_queriesLog, '/query'));
				if (!empty($queries)) {
					$message .= " Query trace (newest to oldest): \n\t";
					$message .= implode("\n\t", array_slice($queries, 0, 5));
				}
				$this->lockLog($message);
			}

			if ($retry < $this->config['lock']['retries']) {
				$result = $this->_execute($sql, $retry + 1);
			} elseif (!empty($this->config['lock']['log'])) {
				$this->lockLog('Failed after ' . number_format($retry) . ' retries');
			}
		} elseif (empty($error) && $retry > 0 && !empty($this->config['lock']['log'])) {
			$this->lockLog('Succeeded after ' . number_format($retry) . ' retries');
		}

		if (empty($error) && !$this->fullDebug && !empty($this->config['lock']['log'])) {
			$this->logQuery($sql);
		}
		return $result;
	}

	/**
	 * Logs a message to the lock log file
	 *
	 * @param string $message Message
	 * @return boolean Success
	 */
	protected function lockLog($message) {
		$message = '['.date('d/m/Y H:i:s') . '] ' . $message . "\n";
		$handle = fopen($this->config['lock']['log'], 'a');
		if (!is_resource($handle)) {
			trigger_error(sprintf('Could not open log file %s', $this->config['lock']['log']), E_USER_WARNING);
			return false;
		}

		fwrite($handle, $message);
		fclose($handle);
		return true;
	}
}
?>