﻿namespace AddisonWesley.Michaelis.EssentialCSharp.Chapter10.Listing10_23
{
    using System;
    using System.IO;
    using System.Linq;
    using System.Runtime.CompilerServices;
    using static ConsoleLogger;

    public static class Program
    {
        public static void Main(string[] args)
        {
            WriteLine("Rozpoczynanie pracy...");
            DoStuff();
            if (args.Any(arg => arg.ToLower() == "-gc"))
            {
                GC.Collect();
                GC.WaitForPendingFinalizers();
            }
            WriteLine("Kończenie pracy...");
        }

        // ...
        public static void DoStuff()
        {
            WriteLine("Rozpoczynanie pracy...");
            SampleUnmanagedResource sampleUnmanagedResource =
                new SampleUnmanagedResource();

            // Używanie obiektu
            // ...

            if (Environment.GetCommandLineArgs().Any(arg => arg.ToLower() == "-dispose"))
            {
                sampleUnmanagedResource.Dispose();
            }

            // ...
            WriteLine("Kończenie pracy...");
        }
    }

    class SampleUnmanagedResource : IDisposable
    {
        public SampleUnmanagedResource(string fileName)
        {
            WriteLine("Rozpoczynanie pracy...", $"{nameof(SampleUnmanagedResource)}.ctor");

            WriteLine("Tworzenie zasobów zarządzanych...", $"{nameof(SampleUnmanagedResource)}.ctor");
            WriteLine("Tworzenie zasobów niezarządzanych...", $"{nameof(SampleUnmanagedResource)}.ctor");

            var weakReferenceToSelf = new WeakReference<IDisposable>(this);
            ProcessExitHandler = (_, __) =>
            {
                WriteLine("Rozpoczynanie pracy...", "ProcessExitHandler");
                if (weakReferenceToSelf.TryGetTarget(out IDisposable? self))
                {
                    self.Dispose();
                }
                WriteLine("Kończenie pracy...", "ProcessExitHandler");
            };
            AppDomain.CurrentDomain.ProcessExit
                += ProcessExitHandler;
            WriteLine("Kończenie pracy...", $"{nameof(SampleUnmanagedResource)}.ctor");
        }

        // Zapisywanie delegata do obsługi zdarzenia ProcessExit, dzięki czemu można  
        // go usunąć, jeśli metoda Dispose() lub Finalize() została już wywołana.
        private EventHandler ProcessExitHandler { get; }

        public SampleUnmanagedResource()
            : this(Path.GetTempFileName()) { }

        ~SampleUnmanagedResource()
        {
            WriteLine("Rozpoczynanie pracy...");
            Dispose(false);
            WriteLine("Kończenie pracy...");
        }

        #region Składowe interfejsu IDisposable
        public void Dispose()
        {
            Dispose(true);
        }
        #endregion
        public void Dispose(bool disposing)
        {
            WriteLine("Rozpoczynanie pracy...");

            // Nie trzeba usuwać własnego zarządzanego obiektu
            // (z finalizatorem), jeśli metoda została wywołana przez finalizator.
            // Powodem jest to, że finalizator takich obiektów zostanie
            // (lub już został) wywołany w przetwarzanej kolejce finalizacji.

            if (disposing)
            {
                WriteLine("Usuwanie zasobów zarządzanych...");

                // Wyrejestrowanie z kolejki finalizacji
                System.GC.SuppressFinalize(this);
            }

            AppDomain.CurrentDomain.ProcessExit -= ProcessExitHandler;

            WriteLine("Usuwanie zasobów niezarządzanych...");

            WriteLine("Kończenie pracy...");
        }
    }

    public static class ConsoleLogger
    {
        public static void WriteLine(string? message = null, [CallerMemberName] string? name = null)
            => Console.WriteLine($"{$"{name}: " }{ message ?? ": Wykonywanie" }");
    }

}