﻿using Microsoft.AspNetCore.Mvc;
using CS7;
using SerwisNorthwind.Repozytoria;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace SerwisNorthwind.Controllers
{
   // adres bazowy: api/klienci
   [Route("api/[controller]")]
   public class KlienciController : Controller
   {
      private IRepozytoriumCustomer repo;
      
      // konstruktor wstrzykuje zarejestrowane repozytorium
      public KlienciController(IRepozytoriumCustomer repo)
      {
         this.repo = repo;
      }

      // GET: api/klienci
      // GET: api/klienci/?kraj=[kraj]
      [HttpGet]
      public async Task<IEnumerable<Customer>> OdczytajKlientow(string kraj)
      {
         if (string.IsNullOrWhiteSpace(kraj))
         {
            return await repo.OdczytajWszystkoAsync();
         }
         else
         {
            return (await repo.OdczytajWszystkoAsync())
            .Where(klient => klient.Country == kraj);
         }
      }

      // GET: api/klienci/[id]
      [HttpGet("{id}", Name = "OdczytajKlienta")]
      public async Task<IActionResult> OdczytajKlienta(string id)
      {
         Customer k = await repo.OdczytajAsync(id);
         if (k == null)
         {
            return NotFound(); // 404 Nie znaleziono
         }
         return new ObjectResult(k); // 200 OK
      }

      // POST: api/klienci
      // BODY: Customer (JSON, XML)
      [HttpPost]
      public async Task<IActionResult> Utworz([FromBody] Customer k)
      {
         if (k == null)
         {
            return BadRequest(); // 400 Błąd żądania
         }
         Customer dodane = await repo.UtworzAsync(k);
         return CreatedAtRoute("OdczytajKlienta", // użyj nazwanej ścieżki
         new { id = dodane.CustomerID.ToLower() }, k); // 201 Utworzono
      }

      // PUT: api/klienci/[id]
      // BODY: Customer (JSON, XML)
      [HttpPut("{id}")]
      public async Task<IActionResult> Aktualizuj(string id, [FromBody] Customer k)
      {
         id = id.ToUpper();
         k.CustomerID = k.CustomerID.ToUpper();
         if (k == null || k.CustomerID != id)
         {
            return BadRequest(); // 400 Błąd żądania
         }
         var istniejace = await repo.OdczytajAsync(id);
         if (istniejace == null)
         {
            return NotFound(); // 404 Nie znaleziono
         }
         await repo.AktualizujAsync(id, k);
         return new NoContentResult(); // 204 Bez treści
      }

      // DELETE: api/klienci/[id]
      [HttpDelete("{id}")]
      public async Task<IActionResult> Usun(string id)
      {
         var istniejace = await repo.OdczytajAsync(id);
         if (istniejace == null)
         {
            return NotFound(); // 404 Nie znaleziono
         }
         bool usuniete = await repo.UsunAsync(id);
         if (usuniete)
         {
            return new NoContentResult(); // 204 Bez treści
         }
         else
         {
            return BadRequest();
         }
      }
   }
}