﻿// [Route], [ApiController], ControllerBase itd.
using Microsoft.AspNetCore.Mvc;
using Northwind.ModeleEncji; // Customer.
using Northwind.WebApi.Repositories; // ICustomerRepository.

namespace Northwind.WebApi.Controllers;

// Adres bazowy: api/klienci
[Route("api/[controller]")]
[ApiController]
public class KlienciController : ControllerBase
{
  private readonly IKlienciRepository _repo;

  // Konstruktor wstrzykuje repozytorium zarejestrowane w pliku Program.cs.
  public KlienciController(IKlienciRepository repo)
  {
    _repo = repo;
  }

  // GET: api/klienci
  // GET: api/klienci/?kraj=[kraj]
  // Ta metoda zawsze zwraca listę klientów (ta może być jednak pusta)
  [HttpGet]
  [ProducesResponseType(200, Type = typeof(IEnumerable<Customer>))]
  public async Task<IEnumerable<Customer>> PobierzKlientow(string? kraj)
  {
    if (string.IsNullOrWhiteSpace(kraj))
    {
      return await _repo.PobierzWszystkoAsync();
    }
    else
    {
      return (await _repo.PobierzWszystkoAsync())
      .Where(klient => klient.Country == kraj);
    }
  }

  // GET: api/klienci/[id]
  [HttpGet("{id}", Name = nameof(PobierzKlienta))] // ścieżka nazwana.
  [ProducesResponseType(200, Type = typeof(Customer))]
  [ProducesResponseType(404)]
  public async Task<IActionResult> PobierzKlienta(string id)
  {
    Customer? k = await _repo.PobierzAsync(id);
    if (k == null)
    {
      return NotFound(); // 404 Resource not found.
    }
    return Ok(k); // 200 OK i dane klienta w ciele wiadomości
  }

  // POST: api/klienci
  // BODY: Customer (JSON, XML)
  [HttpPost]
  [ProducesResponseType(201, Type = typeof(Customer))]
  [ProducesResponseType(400)]
  public async Task<IActionResult> Utworz([FromBody] Customer k)
  {
    if (k == null)
    {
      return BadRequest(); // 400 Bad request.
    }
    Customer? dodanyKlient = await _repo.UtworzAsync(k);
    if (dodanyKlient == null)
    {
      return BadRequest("Nie udało się utworzyć klienta w repozytorium.");
    }
    else
    {
      return CreatedAtRoute( // 201 Created.
      routeName: nameof(PobierzKlienta),
      routeValues: new { id = dodanyKlient.CustomerId.ToLower() },
      value: dodanyKlient);
    }
  }

  // PUT: api/klienci/[id]
  // BODY: Customer (JSON, XML)
  [HttpPut("{id}")]
  [ProducesResponseType(204)]
  [ProducesResponseType(400)]
  [ProducesResponseType(404)]
  public async Task<IActionResult> Aktualizuj(
  string id, [FromBody] Customer k)
  {
    id = id.ToUpper();
    k.CustomerId = k.CustomerId.ToUpper();
    if (k == null || k.CustomerId != id)
    {
      return BadRequest(); // 400 Bad request.
    }
    Customer? istniejacy = await _repo.PobierzAsync(id);
    if (istniejacy == null)
    {
      return NotFound(); // 404 Resource not found.
    }
    await _repo.AktualizujAsync(k);
    return new NoContentResult(); // 204 No content.
  }

  // DELETE: api/klienci/[id]
  [HttpDelete("{id}")]
  [ProducesResponseType(204)]
  [ProducesResponseType(400)]
  [ProducesResponseType(404)]
  public async Task<IActionResult> Usun(string id)
  {
    // Zdefiniuj szczegóły problemu.
    if (id == "zly")
    {
      ProblemDetails szczegolyProblemu = new()
      {
        Status = StatusCodes.Status400BadRequest,
        Type = "https://localhost:5151/klienci/nie-udalo-sie-usunac",
        Title = $"Znaleziono klienta o ID {id}, ale nie udało się go usunąć.",
        Detail = "Więcej szczegółów, takich jak nazwa firmy, kraj itd.",
        Instance = HttpContext.Request.Path
      };
      return BadRequest(szczegolyProblemu); // 400 Bad Request
    }

    Customer? istniejacy = await _repo.PobierzAsync(id);
    if (istniejacy == null)
    {
      return NotFound(); // 404 Resource not found.
    }
    bool? usuniety = await _repo.UsunAsync(id);
    if (usuniety.HasValue && usuniety.Value) // Skrócony AND.
    {
      return new NoContentResult(); // 204 No content.
    }
    else
    {
      return BadRequest( // 400 Bad request.
      $"Znaleziono klienta o ID {id}, ale nie udało się go usunąć.");
    }
  }
}
