﻿using Microsoft.EntityFrameworkCore.ChangeTracking; // EntityEntry<T>.
using Northwind.ModeleEncji; // Customer.
using Microsoft.Extensions.Caching.Memory; // IMemoryCache.
using Microsoft.EntityFrameworkCore; // ToArrayAsync.

namespace Northwind.WebApi.Repositories;

public class KlienciRepository : IKlienciRepository
{
  private readonly IMemoryCache _pamiecPodreczna;

  private readonly MemoryCacheEntryOptions _opcjePamieciPodrecznej = new()
  {
    SlidingExpiration = TimeSpan.FromMinutes(30)
  };

  private KontekstNorthwind _db;

  public KlienciRepository(KontekstNorthwind db,
  IMemoryCache memoryCache)
  {
    _db = db;
    _pamiecPodreczna = memoryCache;
  }

  public async Task<Customer?> UtworzAsync(Customer k)
  {
    k.CustomerId = k.CustomerId.ToUpper(); // Normalizacja do wielkich liter.
                                           // Dodanie do bazy danych za pomocą EF Core.
    EntityEntry<Customer> dodano = await _db.Customers.AddAsync(k);
    int zmienionych = await _db.SaveChangesAsync();
    if (zmienionych == 1)
    {
      // Po zapisaniu w bazie, zachowaj w pamięci podręcznej.
      _pamiecPodreczna.Set(k.CustomerId, k, _opcjePamieciPodrecznej);
      return k;
    }
    return null;
  }

  public Task<Customer[]> PobierzWszystkoAsync()
  {
    return _db.Customers.ToArrayAsync();
  }

  public Task<Customer?> PobierzAsync(string id)
  {
    id = id.ToUpper(); // Normalizowanie do wielkich liter.
                       // Najpierw spróbuj pobrać z pamięci podręcznej.
    if (_pamiecPodreczna.TryGetValue(id, out Customer? zPamieciPodrecznej))
      return Task.FromResult(zPamieciPodrecznej);
    // Jeżeli nie ma w pamięci podręcznej, spróduj pobrać z bazy danych.
    Customer? zBazyDanych = _db.Customers.FirstOrDefault(k => k.CustomerId ==
    id);
    // Jeżeli nie ma w bazie danych, zwróć wartość null.
    if (zBazyDanych is null) return Task.FromResult(zBazyDanych);
    // Jeżeli jest w bazie danych, to zapisz w pamięci podręcznej i zwróć dane klienta.
    _pamiecPodreczna.Set(zBazyDanych.CustomerId, zBazyDanych, _opcjePamieciPodrecznej);
    return Task.FromResult(zBazyDanych)!;
  }

  public async Task<Customer?> AktualizujAsync(Customer k)
  {
    k.CustomerId = k.CustomerId.ToUpper();
    _db.Customers.Update(k);
    int zmienionych = await _db.SaveChangesAsync();
    if (zmienionych == 1)
    {
      _pamiecPodreczna.Set(k.CustomerId, k, _opcjePamieciPodrecznej);
      return k;
    }
    return null;
  }

  public async Task<bool?> UsunAsync(string id)
  {
    id = id.ToUpper();
    Customer? k = await _db.Customers.FindAsync(id);
    if (k is null) return null;
    _db.Customers.Remove(k);
    int zmienionych = await _db.SaveChangesAsync();
    if (zmienionych == 1)
    {
      _pamiecPodreczna.Remove(k.CustomerId);
      return true;
    }
    return null;
  }
}