using Microsoft.AspNetCore.Mvc;
using Northwind.Mvc.Models;
using System.Diagnostics;
using Northwind.ModeleEncji; // KontekstNorthwind.
using Microsoft.EntityFrameworkCore;
using Microsoft.AspNetCore.Authorization; // [Authorize].

namespace Northwind.Mvc.Controllers
{
  public class HomeController : Controller
  {
    private readonly ILogger<HomeController> _logger;
    private readonly KontekstNorthwind _db;
    private readonly IHttpClientFactory _fabrykaKlientow;

    public HomeController(
      ILogger<HomeController> logger,
      KontekstNorthwind db,
      IHttpClientFactory fabrykaKlientow)
    {
      _logger = logger;
      _db = db;
      _fabrykaKlientow = fabrykaKlientow;
    }

    [ResponseCache(Duration = 10 /* sekund */,
      Location = ResponseCacheLocation.Any)]
    public async Task<IActionResult> Index()
    {
      _logger.LogError("To poważny błąd (tylko żartuję!)");
      _logger.LogWarning("To jest Twoje pierwsze ostrzeżenie!");
      _logger.LogWarning("Drugie ostrzeżenie!");
      _logger.LogInformation("Jestem metodą Index kontrolera HomeController.");

      HomeIndexViewModel model = new
      (
        LiczbaOdwiedzin: Random.Shared.Next(1, 1001),
        Kategorie: await _db.Categories.ToListAsync(),
        Produkty: await _db.Products.ToListAsync()
      );

      try
      {
        HttpClient klient = _fabrykaKlientow.CreateClient(
          name: "Northwind.MinimalApi");

        HttpRequestMessage zadanie = new(
          method: HttpMethod.Get, requestUri: "dozrobienia");

        HttpResponseMessage odpowiedz = await klient.SendAsync(zadanie);

        ViewData["dozrobienia"] = await odpowiedz.Content
          .ReadFromJsonAsync<DoZrobienia[]>();
      }
      catch (Exception ex)
      {
        _logger.LogWarning(
          $"Serwis Minimal.WebApi nie odpowiada. Wyjątek: {ex.Message}");

        ViewData["dozrobienia"] = Enumerable.Empty<DoZrobienia>().ToArray();
      }

      return View(model); // Przekaż model do widoku.
    }

    [Route("private")]
    [Authorize(Roles = "Administrators")]
    public IActionResult Privacy()
    {
      return View();
    }

    [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
    public IActionResult Error()
    {
      return View(new ErrorViewModel { RequestId = Activity.Current?.Id ?? HttpContext.TraceIdentifier });
    }

    public async Task<IActionResult> DaneProduktu(int? id,
      string stylalertu = "success")
    {
      ViewData["alertstyle"] = stylalertu;

      if (!id.HasValue)
      {
        return BadRequest("Id produktu musisz podać w ścieżce, na przykład: /Home/DaneProduktu/21");
      }

      Product? model = await _db.Products.Include(p => p.Category)
        .SingleOrDefaultAsync(p => p.ProductId == id);

      if (model is null)
      {
        return NotFound($"Nie znaleziono produktu o ID {id}.");
      }

      return View(model); // Przekaż model do widoku i zwróć wynik.
    }

    // Ta metoda akcji obsłuży żądania GET i inne, zwyjątkiem POST.
    public IActionResult WiazanieModelu()
    {
      return View(); // Strona z formularzem.
    }

    [HttpPost] // Ta metoda akcji obsłuży żądania POST.
    public IActionResult WiazanieModelu(Rzecz rzecz)
    {
      HomeWiazanieModeluViewModel model = new(
        Rzecz: rzecz, 
        MaBledy: !ModelState.IsValid,
        Bledy: ModelState.Values
          .SelectMany(stan => stan.Errors)
          .Select(blad => blad.ErrorMessage)
      );

      return View(model); // Wyświetl rzecz związaną z modelem.
    }

    public IActionResult ProduktyKosztujaceWiecejNiz(decimal? cena)
    {
      if (!cena.HasValue)
      {
        return BadRequest("Musisz podać cenę w zapytaniu, Na przykład: /Home/ProduktyKosztujaceWiecejNiz?cena=50");
      }

      IEnumerable<Product> model = _db.Products
        .Include(p => p.Category)
        .Include(p => p.Supplier)
        .Where(p => p.UnitPrice > cena);

      if (!model.Any())
      {
        return NotFound(
          $"Nie ma produktow droższych niż {cena:C}.");
      }

      ViewData["MaxCena"] = cena.Value.ToString("C");

      return View(model);
    }

    public async Task<IActionResult> Klienci(string kraj)
    {
      string uri;

      if (string.IsNullOrEmpty(kraj))
      {
        ViewData["Tytul"] = "KLienci z całego świata";
        uri = "api/klienci";
      }
      else
      {
        ViewData["Tytul"] = $"Klienci z {kraj}";
        uri = $"api/klienci/?kraj={kraj}";
      }

      HttpClient klient = _fabrykaKlientow.CreateClient(
        name: "Northwind.WebApi");

      HttpRequestMessage zadanie = new(
        method: HttpMethod.Get, requestUri: uri);

      HttpResponseMessage odpowiedz = await klient.SendAsync(zadanie);

      IEnumerable<Customer>? model = await odpowiedz.Content
        .ReadFromJsonAsync<IEnumerable<Customer>>();

      return View(model);
    }

    // GET /Home/DodajKlienta
    public IActionResult DodajKlienta()
    {
      ViewData["Tytul"] = "Dodaj klienta";
      return View();
    }

    // POST /Home/DodajKlienta
    // Obiekt z danymi klienta w ciele żądania.
    [HttpPost]
    public async Task<IActionResult> DodajKlienta(Customer klient)
    {
      HttpClient klientHttp = _fabrykaKlientow.CreateClient(
        name: "Northwind.WebApi");

      HttpResponseMessage odpowiedz = await klientHttp.PostAsJsonAsync(
        requestUri: "api/klienci", value: klient);

      // Opcjonalnie można odczytać dane klienta z dokumentu JSON
      // na przykład, aby przypisać mu identyfikator.
      Customer? model = await odpowiedz.Content
        .ReadFromJsonAsync<Customer>();

      if (odpowiedz.IsSuccessStatusCode)
      {
        TempData["success-message"] = "Utworzono klienta.";
      }
      else
      {
        TempData["error-message"] = "Nie udało się utworzyć klienta.";
      }

      // Wyświetl listę wszystkich klientów, aby sprawdzić, czy nowy został dodany.
      return RedirectToAction("Klienci");
    }

    // GET /Home/UsunKlienta/{IdKlienta}
    public async Task<IActionResult> UsunKlienta(string idKlienta)
    {
      HttpClient klientHttp = _fabrykaKlientow.CreateClient(
        name: "Northwind.WebApi");

      Customer? klient = await klientHttp.GetFromJsonAsync<Customer>(
        requestUri: $"api/klienci/{idKlienta}");

      ViewData["Tytul"] = "Usuń klienta";

      return View(klient);
    }

    // POST /Home/UsunKlienta
    // Id Klienta znajduje się w ciele żądania, np. ALFKI.
    [HttpPost]
    [Route("home/usunklienta")]
    // Metoda akcji musi mieć nazwę inną od metody obsługującej żądania GET
    // ponieważ język C# nie pozwala stosować takich samych sygnatur metod.
    public async Task<IActionResult> UsunKlientaPost(string idKlienta)
    {
      HttpClient klientHttp = _fabrykaKlientow.CreateClient(
        name: "Northwind.WebApi");

      HttpResponseMessage odpowiedz = await klientHttp.DeleteAsync(
        requestUri: $"api/klienci/{idKlienta}");

      if (odpowiedz.IsSuccessStatusCode)
      {
        TempData["success-message"] = "Usunięto dane klienta.";
      }
      else
      {
        TempData["error-message"] = $"Klient o Id {idKlienta} NIE został usunięty.";
      }

      // Wyświetl pełną listę klientów, aby sprawdzić, czy został usunięty.
      return RedirectToAction("Klienci");
    }

    public async Task<IActionResult> DaneKategorii(int? id)
    {
      if (!id.HasValue)
      {
        return BadRequest("Musisz podać ID w ramach ścieżki, na przykład: /Home/DaneKategorii/6");
      }

      Category? model = await _db.Categories.Include(p => p.Products)
        .SingleOrDefaultAsync(p => p.CategoryId == id);

      if (model is null)
      {
        return NotFound($"Nie znaleziono kategorii o ID {id}.");
      }

      return View(model); // przekaż model do widoku i zwróć wynik.
    }
  }
}