/*
 * DAC.c
 *
 * Created: 2013-09-14 17:16:30
 *  Author: tmf
 */

#include <stdbool.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#define GET_FAR_ADDRESS(var)                  \
({                                            \
	__uint24 tmp;                             \
	__asm__ __volatile__(                     \
	"ldi	%A0, lo8(%1)"           "\n\t"    \
	"ldi	%B0, hi8(%1)"           "\n\t"    \
	"ldi	%C0, hh8(%1)"           "\n\t"    \
	: "=d" (tmp) :	"p"  (&(var)));           \
	tmp;                                      \
})

#define BUFFER_SIZE  512     //Wielko tymczasowego bufora na prbki (uyta jest dwukrotnie wiksza)

extern __memx const uint8_t _binary____intro8kHz8PWMHL_raw_end;
extern const __uint24 _binary____intro8kHz8PWMHL_raw_size;
extern __memx const uint8_t _binary____intro8kHz8PWMHL_raw_start;

extern __memx const uint8_t _binary____intro8kHz16PWMu_raw_end;
extern const __uint24 _binary____intro8kHz16PWMu_raw_size;
extern __memx const uint8_t _binary____intro8kHz16PWMu_raw_start;

uint8_t samplebuffer[2][BUFFER_SIZE];  //Bufor na odgrywane prbki
__uint24 srcaddr;                      //Adres prbek dwikowych
__uint24 samplesize;                   //Dugo odtwarzanej prbki w bajtach
volatile _Bool isPlaying;              //Czy aktualnie jest co odtwarzane

void TMF_memcpy_PF(void *dst, __uint24 src, size_t len)
{
	   asm(
	         "in __tmp_reg__, %1" "\n\t"
	         "out %1, %C0"   "\n\t"
	         "movw r30, %0"  "\n\t"
		  "L_dl1%=:" "\n\t"
	           "elpm %A0, Z+"  "\n\t"
			   "st X+, %A0"  "\n\t"
			   "SBIW %3, 1"  "\n\t"
			   "brne L_dl1%=" "\n\t"
	         "out %1, __tmp_reg__"
	         : : "r" (src),
	           "I" (_SFR_IO_ADDR(RAMPZ)), "x" (dst), "w" (len)
	         : "r30", "r31", "memory"
	     );
}

void DAC_init()
{
	//DACB.CTRLC=DAC_REFSEL_INT1V_gc | DAC_LEFTADJ_bm;                //Wewn. napicie ref. 1 V
	DACB.CTRLC=DAC_REFSEL_AVCC_gc | DAC_LEFTADJ_bm;                //Wewn. napicie ref. 1 V
	DACB.TIMCTRL=DAC_CONINTVAL_32CLK_gc | DAC_REFRESH_128CLK_gc;           //CLK/32
	DACB.CTRLB=DAC_CH0TRIG_bm;               //Konwersja wyzwalana zdarzeniem w kanale CH0
	DACB.EVCTRL=DAC_EVSEL_0_gc;              //EvCh 0
	EVSYS_CH0MUX=EVSYS_CHMUX_TCC0_OVF_gc; //taktowanie evch0 z przepenienia timera C0
	DACB.CTRLA=DAC_ENABLE_bm | DAC_CH0EN_bm; //Wcz DAC, kana 0 routowany na PB2
}

ISR(DMA_CH0_vect)
{
	DMA_INTFLAGS=DMA_CH0TRNIF_bm; //Skasuj flag przerwania
	if(((DMA_CH0_CTRLA & DMA_CH_ENABLE_bm) == 0) && ((DMA_CH1_CTRLA & DMA_CH_ENABLE_bm) == 0))
	{
		isPlaying=false;  //zakoczylimy odtwarzanie dwiku
		return;
	}
	if(DMA_CH0_REPCNT == 1) DMA_CH0_TRFCNT=samplesize % BUFFER_SIZE;
	TMF_memcpy_PF(&samplebuffer[0][0], srcaddr, BUFFER_SIZE);
	srcaddr+=BUFFER_SIZE;
}

ISR(DMA_CH1_vect)
{
	DMA_INTFLAGS=DMA_CH1TRNIF_bm; //Skasuj flag przerwania
	if(((DMA_CH0_CTRLA & DMA_CH_ENABLE_bm) == 0) && ((DMA_CH1_CTRLA & DMA_CH_ENABLE_bm) == 0))
	{
		isPlaying=false;  //zakoczylimy odtwarzanie dwiku
		return;
	}
	if(DMA_CH0_REPCNT == 1) DMA_CH0_TRFCNT=samplesize % BUFFER_SIZE;
	TMF_memcpy_PF(&samplebuffer[1][0], srcaddr, BUFFER_SIZE);
	srcaddr+=BUFFER_SIZE;
}

void Timer_init(uint16_t samplerate)  //Generuje zdarzenia wywoujce konwersj DAC i przeczanie buforw
{
	//TCC0 - generuje zdarzenia wywoujce konwersj (okrela samplerate)
	TCC0.PER=F_CPU/(samplerate - 1); //Wygeneruj zdarzenia o odpowiedniej czstotliwoci
	TCC0.CTRLA=TC_CLKSEL_DIV1_gc;
}

void Amplifier_init()
{
	PORTQ.OUTSET=PIN3_bm;  //Aktywacja wzmacniacza TPA0253
	PORTQ.DIRSET=PIN3_bm;
	PORTB.OUTCLR=PIN2_bm;
	PORTB.DIRSET=PIN2_bm;  //Pin DACB0 jako wyjcie
}

void Snd_init(__uint24 smpaddr, __uint24 smpsize, uint16_t bitrate, _Bool HiRes)
{
	void DMA_init()
	{
		void DMA_CH_Cfg(DMA_CH_t *CH, uint16_t srcaddr, uint8_t repcnt)
		{
			CH->TRIGSRC=DMA_CH_TRIGSRC_DACB_CH0_gc;          //Zdarzeniem wyzwalajcym jest pusty rejestr danych kanau CH0 DACA
			CH->TRFCNT=BUFFER_SIZE;                          //Blok ma dugo rwn buforowi prbek
			CH->REPCNT=repcnt;
			CH->SRCADDR0=srcaddr & 0xFF;
			CH->SRCADDR1=srcaddr >> 8;
			CH->SRCADDR2=0;                                  //rdem danych jest wskazany bufor
			CH->CTRLB=DMA_CH_TRNINTLVL_LO_gc;                //Generuj przerwaie koca transakcji
			if(HiRes)
			{
				CH->DESTADDR0=(uint16_t)(&DACB.CH0DATAL) & 0xFF; //Dane wpisujemy do rejestru DATA ukadu DAC
				CH->DESTADDR1=(uint16_t)(&DACB.CH0DATAL) >> 8;
				CH->DESTADDR2=0;
				CH->ADDRCTRL=DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTDIR_INC_gc | DMA_CH_DESTRELOAD_BURST_gc; //Zwiekszamy adres rda i przeznaczenia, reload adresu co blok, adresu docelowego co transfer
				CH->CTRLA=DMA_CH_ENABLE_bm | DMA_CH_REPEAT_bm | DMA_CH_BURSTLEN_2BYTE_gc | DMA_CH_SINGLE_bm;  //Kana 0 w trybie powtarzania, dugo transferu 2 bajty, single shot
			} else
			{
				CH->DESTADDR0=(uint16_t)(&DACB.CH0DATAH) & 0xFF; //Dane wpisujemy do rejestru DATA ukadu DAC
				CH->DESTADDR1=(uint16_t)(&DACB.CH0DATAH) >> 8;
				CH->DESTADDR2=0;
				CH->ADDRCTRL=DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTDIR_FIXED_gc | DMA_CH_DESTRELOAD_NONE_gc; //Zwiekszamy adres rda, reload adresu co blok, adres docelowy bez zmian
				CH->CTRLA=DMA_CH_ENABLE_bm | DMA_CH_REPEAT_bm | DMA_CH_BURSTLEN_1BYTE_gc | DMA_CH_SINGLE_bm;  //Kana 0 w trybie powtarzania, dugo transferu 1 bajt, single shot
			}
		}

		uint16_t repcnt=(smpsize + BUFFER_SIZE) / BUFFER_SIZE;
		DMA_CH_Cfg(&DMA.CH0, (uint16_t)&samplebuffer[0][0], (repcnt & 1) ? (repcnt/2+1) : (repcnt/2));      //Adres pierwszego bufora
		DMA_CH_Cfg(&DMA.CH1, (uint16_t)&samplebuffer[1][0], repcnt/2);                                      //Adres drugiego bufora
		DMA.CTRL=DMA_ENABLE_bm | DMA_DBUFMODE_CH01_gc;  //Odblokuj kontroler DMA, round robin, podwjne buforowanie kanay 0 & 1
	}

	samplesize=smpsize;  //Dugo odtwarzanej prbki w bajtach
	srcaddr=smpaddr;    //Adres prbki
	TMF_memcpy_PF(samplebuffer, srcaddr, BUFFER_SIZE*2);   //Zainicjuj bufor prbek w SRAM
	srcaddr+=BUFFER_SIZE*2;                            //Adres dalszej czci prbek

	DMA_init();
	Timer_init(bitrate);  //Zainicjuj timer z odpowiednim samplerate
	DAC_init();           //Zainicjuj transfer DMA prbek
	isPlaying=true;
}

int main(void)
{
	Amplifier_init();
	isPlaying=false;
	PMIC_CTRL=PMIC_LOLVLEN_bm;  //Odblokuj przerwania najniszego poziomu
	sei();

	PORTD_PIN0CTRL=PORT_OPC_PULLUP_gc; //Wczamy pullup na SW0
	PORTD_PIN1CTRL=PORT_OPC_PULLUP_gc; //Wczamy pullup na SW1

    while(1)
    {
       if(((PORTD_IN & PIN0_bm) == 0) && (isPlaying == false)) Snd_init(GET_FAR_ADDRESS(_binary____intro8kHz8PWMHL_raw_start), GET_FAR_ADDRESS(_binary____intro8kHz8PWMHL_raw_size), 8000, false);  //Zagraj prbk 8kHz/8 bitw
	   if(((PORTD_IN & PIN1_bm) == 0) && (isPlaying == false)) Snd_init(GET_FAR_ADDRESS(_binary____intro8kHz16PWMu_raw_start), GET_FAR_ADDRESS(_binary____intro8kHz16PWMu_raw_size), 8000, true);  //Zagraj prbk 16kHz/16 bitw
    }
}