/*
 * DAC_1CHDMA.c
 *
 * Created: 2013-09-23 19:05:49
 *  Author: tmf
 */


#include <stdbool.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#define GET_FAR_ADDRESS(var)                  \
({                                            \
	__uint24 tmp;                             \
	__asm__ __volatile__(                     \
	"ldi	%A0, lo8(%1)"           "\n\t"    \
	"ldi	%B0, hi8(%1)"           "\n\t"    \
	"ldi	%C0, hh8(%1)"           "\n\t"    \
	: "=d" (tmp) :	"p"  (&(var)));           \
	tmp;                                      \
})

#define BUFFER_SIZE  512     //Wielko tymczasowego bufora na prbki (uyta jest dwukrotnie wiksza)

extern __memx const uint8_t _binary____intro8kHz8PWMHL_raw_end;
extern const __uint24 _binary____intro8kHz8PWMHL_raw_size;
extern __memx const uint8_t _binary____intro8kHz8PWMHL_raw_start;

extern __memx const uint8_t _binary____intro8kHz16PWMu_raw_end;
extern const __uint24 _binary____intro8kHz16PWMu_raw_size;
extern __memx const uint8_t _binary____intro8kHz16PWMu_raw_start;

#define Snd_NoPlaying      0        //Nic nie jest odgrywane
#define Snd_Playing8Bit    1        //Odtwarzamy 8-bitowe prbki
#define Snd_Playing16Bit   2        //Odtwarzamy 16-bitowe prbki
#define Snd_Finishing      4        //Zaraz zakoczymy odtwarzanie

uint8_t samplebuffer[2][BUFFER_SIZE];  //Bufor na odgrywane prbki
__uint24 srcaddr;                      //Adres prbek dwikowych
__uint24 samplesize;                   //Dugo odtwarzanej prbki w bajtach
volatile uint8_t isPlaying;            //Stan odtwarzacza

void TMF_memcpy_PF(void *dst, __uint24 src, size_t len)
{
	asm(
	"in __tmp_reg__, %1" "\n\t"
	"out %1, %C0"   "\n\t"
	"movw r30, %0"  "\n\t"
	"L_dl1%=:" "\n\t"
	"elpm %A0, Z+"  "\n\t"
	"st X+, %A0"  "\n\t"
	"SBIW %3, 1"  "\n\t"
	"brne L_dl1%=" "\n\t"
	"out %1, __tmp_reg__"
	: : "r" (src),
	"I" (_SFR_IO_ADDR(RAMPZ)), "x" (dst), "w" (len)
	: "r30", "r31", "memory"
	);
}

void DAC_init()
{
	//DACB.CTRLC=DAC_REFSEL_INT1V_gc | DAC_LEFTADJ_bm;                //Wewn. napicie ref. 1 V
	DACB.CTRLC=DAC_REFSEL_AVCC_gc | DAC_LEFTADJ_bm;                //Wewn. napicie ref. 1 V
	DACB.TIMCTRL=DAC_CONINTVAL_32CLK_gc | DAC_REFRESH_128CLK_gc;           //CLK/32
	DACB.CTRLB=DAC_CH0TRIG_bm;               //Konwersja wyzwalana zdarzeniem w kanale CH0
	DACB.EVCTRL=DAC_EVSEL_0_gc;              //EvCh 0
	EVSYS_CH0MUX=EVSYS_CHMUX_TCC0_OVF_gc; //taktowanie evch0 z przepenienia timera C0
	DACB.CTRLA=DAC_ENABLE_bm | DAC_CH0EN_bm; //Wcz DAC, kana 0 routowany na PB2
}

//Wywoywane dla pierwszej poowy bufora
ISR(TCC1_OVF_vect)
{
	if(samplesize > BUFFER_SIZE)
	{
		samplesize-=BUFFER_SIZE;
	} else
	{
		//Koczymy zabaw
		if((samplesize == 0) && (isPlaying & Snd_Finishing))    //Nie ma nic do przesania
		{
			TCC0.CTRLA=TC_CLKSEL_OFF_gc;
			isPlaying=Snd_NoPlaying;
			return;
		}

		if(isPlaying & Snd_Playing8Bit) TCC1_PER=samplesize; else TCC1_PER=samplesize/2;
		samplesize=0;          //Ostatnie przesanie
		isPlaying|=Snd_Finishing;
	}
	TMF_memcpy_PF(&samplebuffer[1][0], srcaddr, BUFFER_SIZE);
	srcaddr+=BUFFER_SIZE;
}

//Wywoywane dla drugiej poowy bufora
ISR(TCC1_CCA_vect)
{
	if(samplesize > BUFFER_SIZE)
	{
		samplesize-=BUFFER_SIZE;
	} else
	{
		//Koczymy zabaw
		if((samplesize == 0) && (isPlaying & Snd_Finishing))    //Nie ma nic do przesania
		{
			TCC0.CTRLA=TC_CLKSEL_OFF_gc;
			isPlaying=Snd_NoPlaying;
			return;
		}

		if(isPlaying & Snd_Playing8Bit) TCC1_PERBUF=samplesize; else TCC1_PERBUF=samplesize/2;
		samplesize=0;          //Ostatnie przesanie
		isPlaying|=Snd_Finishing;
	}
	TMF_memcpy_PF(&samplebuffer[0][0], srcaddr, BUFFER_SIZE);
	srcaddr+=BUFFER_SIZE;
}

void Amplifier_init()
{
	PORTQ.OUTSET=PIN3_bm;  //Aktywacja wzmacniacza TPA0253
	PORTQ.DIRSET=PIN3_bm;
	PORTB.OUTCLR=PIN2_bm;
	PORTB.DIRSET=PIN2_bm;  //Pin DACB0 jako wyjcie
}

void Snd_init(__uint24 smpaddr, __uint24 smpsize, uint16_t bitrate, _Bool HiRes)
{
	void Timer_init(uint16_t samplerate)  //Generuje zdarzenia wywoujce konwersj DAC i przeczanie buforw
	{
		//TCC0 - generuje zdarzenia wywoujce konwersj (okrela samplerate)
		TCC0.PER=F_CPU/(samplerate - 1); //Wygeneruj zdarzenia o odpowiedniej czstotliwoci
		TCC0.CTRLB=TC_WGMODE_NORMAL_gc;
		TCC0.CTRLA=TC_CLKSEL_DIV1_gc;

		if(HiRes)
		{
			TCC1.PER=BUFFER_SIZE - 1;       //Timer zlicza a do koca bufora
			TCC1.CCA=BUFFER_SIZE/2;         //Po poowie bufora generujemy przerwanie
		} else
		{
			TCC1.PER=BUFFER_SIZE*2 - 1;       //Timer zlicza a do koca bufora
			TCC1.CCA=BUFFER_SIZE - 1;         //Po poowie bufora generujemy przerwanie
		}
		TCC1.INTCTRLA=TC_OVFINTLVL_LO_gc; //Generuj przerwanie nadmiaru timera
		TCC1.INTCTRLB=TC_CCAINTLVL_LO_gc; //i przerwanie porwnania CCA
		TCC1.CTRLA=TC_WGMODE_NORMAL_gc;
		TCC1.CTRLA=TC_CLKSEL_EVCH0_gc;    //Wszystkim steruje timer TCC0
	}

	void DMA_init()
	{
		DMA.CH0.CTRLA=DMA_CH_RESET_bm;                       //Skasuj poprzedni transfer
		while((DMA.CH0.CTRLA & DMA_CH_ENABLE_bm) == DMA_CH_ENABLE_bm);
		DMA.CH0.TRIGSRC=DMA_CH_TRIGSRC_DACB_CH0_gc;          //Zdarzeniem wyzwalajcym jest pusty rejestr danych kanau CH0 DACA
		DMA.CH0.TRFCNT=BUFFER_SIZE *2;                       //Blok ma dugo rwn buforowi prbek
		DMA.CH0.REPCNT=(smpsize + BUFFER_SIZE*2) / (BUFFER_SIZE*2);
		DMA.CH0.SRCADDR0=(uint16_t)&samplebuffer & 0xFF;
		DMA.CH0.SRCADDR1=(uint16_t)&samplebuffer >> 8;
		DMA.CH0.SRCADDR2=0;                                  //rdem danych jest wskazany bufor
		if(HiRes)
		{
			DMA.CH0.DESTADDR0=(uint16_t)(&DACB.CH0DATAL) & 0xFF; //Dane wpisujemy do rejestru DATA ukadu DAC
			DMA.CH0.DESTADDR1=(uint16_t)(&DACB.CH0DATAL) >> 8;
			DMA.CH0.DESTADDR2=0;
			DMA.CH0.ADDRCTRL=DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTDIR_INC_gc | DMA_CH_DESTRELOAD_BURST_gc; //Zwiekszamy adres rda i przeznaczenia, reload adresu co blok, adresu docelowego co transfer
			DMA.CH0.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_REPEAT_bm | DMA_CH_BURSTLEN_2BYTE_gc | DMA_CH_SINGLE_bm;  //Kana 0 w trybie powtarzania, dugo transferu 2 bajty, single shot
		} else
		{
			DMA.CH0.DESTADDR0=(uint16_t)(&DACB.CH0DATAH) & 0xFF; //Dane wpisujemy do rejestru DATA ukadu DAC
			DMA.CH0.DESTADDR1=(uint16_t)(&DACB.CH0DATAH) >> 8;
			DMA.CH0.DESTADDR2=0;
			DMA.CH0.ADDRCTRL=DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTDIR_FIXED_gc | DMA_CH_DESTRELOAD_NONE_gc; //Zwiekszamy adres rda, reload adresu co blok, adres docelowy bez zmian
			DMA.CH0.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_REPEAT_bm | DMA_CH_BURSTLEN_1BYTE_gc | DMA_CH_SINGLE_bm;  //Kana 0 w trybie powtarzania, dugo transferu 1 bajt, single shot
		}
		DMA.CTRL=DMA_ENABLE_bm;  //Odblokuj kontroler DMA, round robin
	}

	samplesize=smpsize - BUFFER_SIZE*2;  //Dugo odtwarzanej prbki w bajtach minus to co skopiujemy do bufora
	srcaddr=smpaddr;    //Adres prbki
	TMF_memcpy_PF(samplebuffer, srcaddr, BUFFER_SIZE*2);   //Zainicjuj bufor prbek w SRAM
	srcaddr+=BUFFER_SIZE*2;                            //Adres dalszej czci prbek

	DMA_init();
	Timer_init(bitrate);  //Zainicjuj timer z odpowiednim samplerate
	DAC_init();           //Zainicjuj transfer DMA prbek
	isPlaying=HiRes ? Snd_Playing16Bit : Snd_Playing8Bit;
}

int main(void)
{
	Amplifier_init();
	isPlaying=false;
	PMIC_CTRL=PMIC_LOLVLEN_bm;  //Odblokuj przerwania najniszego poziomu
	sei();

	PORTD_PIN0CTRL=PORT_OPC_PULLUP_gc; //Wczamy pullup na SW0
	PORTD_PIN1CTRL=PORT_OPC_PULLUP_gc; //Wczamy pullup na SW1

	while(1)
	{
		if(((PORTD_IN & PIN0_bm) == 0) && (isPlaying == Snd_NoPlaying)) Snd_init(GET_FAR_ADDRESS(_binary____intro8kHz8PWMHL_raw_start), GET_FAR_ADDRESS(_binary____intro8kHz8PWMHL_raw_size), 8000, false);  //Zagraj prbk 8kHz/8 bitw
		if(((PORTD_IN & PIN1_bm) == 0) && (isPlaying == Snd_NoPlaying)) Snd_init(GET_FAR_ADDRESS(_binary____intro8kHz16PWMu_raw_start), GET_FAR_ADDRESS(_binary____intro8kHz16PWMu_raw_size), 8000, true);  //Zagraj prbk 16kHz/16 bitw
	}
}