/**
 * (c)2013 Tomasz Francuz
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 *    Atmel microcontroller product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <avr/interrupt.h>
#include <asf.h>
#include "conf_usb.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "ADCOffset.h"

//Makro umieszczajce zadany acuch w przestrzeni adresowej __flash
#define PGM_STR(X) ((const __flash char[]) { X })

typedef void (*CmdHandler)(char *param);

typedef struct
{
	const char __flash *cmd;                //Wskanik do polecenia w formacie ASCIIZ
	const CmdHandler  Handler;              //Funkcja realizujca polecenie
} Command;

void Cmd_info(char *param);     //Wywietla dane z ADC

//Lista rozpoznawanych polece
const __flash Command __flash Polecenia[]={{PGM_STR("info"), Cmd_info}};

int main(void)
{
	void InterpretCommand(char *cmdline)
	{
		uint8_t indeks;
		char *cmd=strtok(cmdline, " \r\n"); //Wydziel polecenie z przekazanej linii
		uint8_t max_indeks=sizeof(Polecenia)/sizeof(Polecenia[0]);
		for(indeks=0; indeks<max_indeks; indeks++)
		{
			if(strcmp_P(cmd, Polecenia[indeks].cmd)==0) //Przeszukaj list polece
			{
				Polecenia[indeks].Handler(cmdline);   //Wywoaj funkcj obsugi przesanego polecenia
				break;
			}
		}
		if(indeks == max_indeks)  //Jeli polecenie nieznane...
		{
			udi_cdc_write_buf(cmd, strlen(cmd));
			udi_cdc_write_buf(" - nieznane polecenie\r\n", 22); //Bd
		}
	}

	PMIC.CTRL = PMIC_LOLVLEN_bm | PMIC_MEDLVLEN_bm | PMIC_HILVLEN_bm; //Wczamy wszystkie poziomy przerwa (potrzebujemy tylko najniszy)
	sei();

	sysclk_init();
	udc_start();

	while (1)
	{
		char Bufor[256];   //Bufor na odebranie linii
		uint8_t indeks=0;

		while(1)
		 {
			if(udi_cdc_is_rx_ready())
			{
				Bufor[indeks] = udi_cdc_getc();
				if(Bufor[indeks] == '\n')
				{
					if((indeks >= 1) && (Bufor[indeks-1] == '\r')) --indeks;
					Bufor[indeks]=0; //Wpisz kocowy znak NUL
					break;           //Odebrano pen lini
				}
				++indeks;
			}
		 }
		InterpretCommand(Bufor);  //Zdekoduj przesane polecenie
	}
}

void main_suspend_action(void)
{
//Interfejs wchodzi w tryb powerdown
}

void main_resume_action(void)
{
//Wybudzenie interfejsu
}

void main_sof_action(void)
{
//	Odebrano ramk
}

bool main_cdc_enable(uint8_t port)
{
	// Otwarto port CDC
	return true;
}

void main_cdc_disable(uint8_t port)
{
	// Zamknito port CDC
}

void main_cdc_set_dtr(uint8_t port, bool b_enable)
{
	if (b_enable) {
		// Host otwar port COM
	}else{
		// Host zamkn port COM
	}
}

void uart_rx_notify(uint8_t port)
{
	//Opcjonalnie powiadomienie o nadejciu znaku
}

void uart_config(uint8_t port, usb_cdc_line_coding_t * cfg)
{
	//Konfiguracja w odpowiedzi na otwarcie CDC
}

void udi_cdc_write_txt(char *txt)
{
	udi_cdc_write_buf(txt, strlen(txt));
}


void Cmd_info(char *param)
{
	char bufor[255];
	ADC_meas_Vcc_Config();                  //Ustaw ADC na pomiar Vcc/10 z wewntrzn referencj
	int16_t res_signed=ADC_GetResults();    //Pobierz wynik pomiaru dla trybu signed
	ADCA.CTRLB&=~ADC_CONMODE_bm;            //Tryb unsigned
	uint16_t res_unsigned=ADC_GetResults(); //Pobierz wynik pomiaru dla trybu unsigned

	sprintf(bufor, "Offset ADC dla trybu unsigned:%u\r\n", res_unsigned-(res_signed<<1));
	udi_cdc_write_buf(bufor, strlen(bufor));
	sprintf(bufor, "Offset ADC dla trybu unsigned przy zwartym PINA1 do masy:%u\r\n", ADC_Meas_OffsetError_Unsigned_Mode());
	udi_cdc_write_buf(bufor, strlen(bufor));
	sprintf(bufor, "Offset ADC dla trybu roznicowego:%d\r\n", ADC_Meas_OffsetError_Diff_Mode());
	udi_cdc_write_buf(bufor, strlen(bufor));
	sprintf(bufor, "Temperatura chipu:%u stopni\r\n", GetTemperature());
	udi_cdc_write_buf(bufor, strlen(bufor));
	sprintf(bufor, "Napiecie zasilania XMEGA [mV]:%u\r\n", (uint16_t)((res_signed*10000UL)/2048)); //10000 wynika z pomiaru Vcc/10 i wyraenia wyniku w mV (1000)
	udi_cdc_write_buf(bufor, strlen(bufor));
}
