/*
 * mp3_sinetest.c
 *
 * Created: 2013-10-28 20:52:52
 *  Author: tmf
 */


#include <avr/io.h>
#include <stdbool.h>
#include <util/delay.h>
#include "VS1003B.h"

//Definicje pinw za pomoc ktrych podczony jest VS1003B
#define VS_DREQ    PIN3_bm
#define VS_xDCS    PIN2_bm
#define VS_xCS     PIN1_bm
#define VS_RESET   PIN4_bm
#define VS_TxD     PIN7_bm
#define VS_SCK     PIN5_bm
#define VS_RxD     PIN6_bm

#define VS_PORT    PORTC      //Port do ktrego jest podczony VS1003B
#define VS_USART   USARTC1    //USART przez ktry nastuje komunikacja

#define VS_SPI_CLK 1000000UL   //Taktowanie SPI dla VS1003B

void SPI_VS_init()
{
	VS_PORT.OUTSET=VS_xCS | VS_xDCS | VS_RESET;
	VS_PORT.DIRSET=VS_xDCS | VS_xCS | VS_RESET | VS_TxD | VS_SCK;   //Piny wyjciowe interfejsu
	VS_USART.CTRLC=USART_CMODE_MSPI_gc;             //Master SPI, MSB first, dane prbokwane na zboczu narastajcym

	#if F_CPU > VS_SPI_CLK
	VS_USART.BAUDCTRLA=F_CPU/VS_SPI_CLK/2 - 1;        //Oblicz taktowanie SPI
	#else
	VS_USART.BAUDCTRLA=0;
	#endif

	VS_USART.BAUDCTRLB=0;   //Taktowanie FCLK/2
	VS_USART.CTRLB=USART_RXEN_bm | USART_TXEN_bm;   //Odblokuj nadajnik i odbiornik
}

uint8_t SPI_VS_io(uint8_t data)
{
	VS_USART.DATA=data;
	while(!(VS_USART.STATUS & USART_TXCIF_bm));
	VS_USART.STATUS=USART_TXCIF_bm;    //Skasuj flag
	return VS_USART.DATA;
}

uint16_t SPI_VS_io_word(uint16_t word)
{
	uint16_t ret=SPI_VS_io(word >> 8) << 8;  //Wylij bardziej znaczcy bajt
	ret+=SPI_VS_io(word & 0xff);             //Wylij mniej znaczcy bajt
	return ret;
}

//Zresetuj koprocesor, zwr false, jeli ukad nie odpowiada
_Bool VS1003_Reset()
{
	VS_PORT.OUTCLR=VS_RESET;  //Reset ukadu
	_delay_ms(1);
	VS_PORT.OUTSET=VS_RESET;

	uint8_t del=255;
	while(((VS_PORT.IN & VS_DREQ) == 0) && (del))   //Czekamy na gotowo ukadu
	{
		_delay_us(100);
		del--;
	}
	return (del != 0);     //Zwr false jeli ukad nie odpowiedzia na RESET
}

//Dostp do interfejsu polece VS1003
static inline void VS1003_xCS_Enable(_Bool en)
{
	if(en)
	{
		VS_PORT.OUTSET=VS_xDCS;    //Deaktywuj xDCS
		VS_PORT.OUTCLR=VS_xCS;     //Aktywuj xCS
	} else VS_PORT.OUTSET=VS_xCS;
}

//Dostp do interfejsu danych VS1003
static inline void VS1003_xDCS_Enable(_Bool en)
{
	if(en)
	{
		VS_PORT.OUTSET=VS_xCS;    //Deaktywuj xCS
		VS_PORT.OUTCLR=VS_xDCS;   //Aktywuj xDCS
	} else VS_PORT.OUTSET=VS_xDCS;
}

static inline void VS_1003_DREQ_High()
{
	while((VS_PORT.IN & VS_DREQ) == 0);
}

void VS1003_WriteReg(uint8_t cmd, uint16_t value)
{
	VS1003_xCS_Enable(true);       //Aktywuj DCI
	SPI_VS_io(VS_WRITE_CMD);
	SPI_VS_io(cmd);
	SPI_VS_io_word(value);         //Domylna konfiguracja
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();           //Zaczekaj na koniec realizacji polecenia
}

uint16_t VS1003_ReadReg(uint8_t cmd)
{
	VS1003_xCS_Enable(true);       //Aktywuj DCI
	SPI_VS_io(VS_READ_CMD);
	SPI_VS_io(cmd);
	uint16_t ret=SPI_VS_io_word(0);       
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();           //Zaczekaj na koniec realizacji polecenia
	return ret;
}

void VS1003_SineTest()
{
	VS1003_xCS_Enable(true);
	VS1003_WriteReg(VS_MODE, *(uint16_t*)&((VS_SCI_MODE){.SM_SDINEW=true, .SM_TESTS=true}));
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();

	VS1003_xDCS_Enable(true);
	SPI_VS_io_word(0x53ef);
	SPI_VS_io_word(0x6e7e);
	SPI_VS_io_word(0x0000);
	SPI_VS_io_word(0x0000);
	VS1003_xDCS_Enable(false);
}

void GPIOR0_init()
{
	VS1003_xCS_Enable(true);                   //Aktywuj DCI
	VS1003_WriteReg(VS_WRAMADDR, VS_GPIO_DIR);
	VS1003_WriteReg(VS_WRAM, 0x0001);          //Wszystkie rejestry s wyjciami
	VS1003_xCS_Enable(false);                  //Deaktywuj DCI
}

void GPIOR0_set(_Bool onoff)
{
	VS1003_xCS_Enable(true);                   //Aktywuj DCI
	VS1003_WriteReg(VS_WRAMADDR, VS_GPIO_OUT);
	VS1003_WriteReg(VS_WRAM, onoff);           //A ich stan jest wysoki
	VS1003_xCS_Enable(false);                  //Deaktywuj DCI
}

_Bool VS1003_init()
{
	SPI_VS_init();                            //Inicjalizacja interfejsu
	if(VS1003_Reset() == false) return false; //Zresetuj ukad
	return true;
}


int main(void)
{
	VS1003_init();
	VS1003_SineTest();   //Testuj ukad
	GPIOR0_init();

    while(1)
    {
         GPIOR0_set(true);
		 _delay_ms(1000);
		 GPIOR0_set(false);
		 _delay_ms(1000);
    }
}