/*
 * VS1003B.c
 *
 * Created: 2013-10-31 10:19:08
 *  Author: tmf
 */

#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <stddef.h>
#include "VS1003B.h"
#include "SPI.h"
#include "ffconf.h"
#include "TransactSPI.h"

uint32_t VS_Buffer_ptr;                      //Wskanik na aktualnie odczytywany bajt w buforze

volatile VS_States VS_Status=VS_NoPlaying;   //Co aktualnie robimy
FIL       sound_file;                        //Plik zawierajcy muzyk do odtworzenia
uint8_t VS_Volume;                           //Gono odtwarzanego dwiku


void VS1003_DMA_init();

void SPI_VS_init(uint16_t clk)
{
	VS_PORT.OUTSET=VS_xCS | VS_xDCS | VS_RESET;
	VS_PORT.DIRSET=VS_xDCS | VS_xCS | VS_RESET | VS_TxD | VS_SCK;   //Piny wyjciowe interfejsu
	VS_DREQCTRL=PORT_ISC_LEVEL_gc | PORT_INVEN_bm;                 //Pin DREQ - zdarzenie generuje poziom niski
	VS_USART.CTRLC=USART_CMODE_MSPI_gc;             //Master SPI, MSB first, dane prbokwane na zboczu narastajcym

	VS_USART.BAUDCTRLA=F_CPU/clk/1000 - 1;
	VS_USART.BAUDCTRLB=0;

	VS_USART.CTRLB=USART_RXEN_bm | USART_TXEN_bm;   //Odblokuj nadajnik i odbiornik
}

//Zresetuj koprocesor, zwr false, jeli ukad nie odpowiada
_Bool VS1003_Reset()
{
	VS_PORT.OUTCLR=VS_RESET;  //Reset ukadu
	_delay_ms(1);
	VS_PORT.OUTSET=VS_RESET;

	uint8_t del=255;
	while(((VS_PORT.IN & VS_DREQ) == VS_DREQ) && (del))   //Czekamy na gotowo ukadu
	{
		_delay_us(100);
		del--;
	}
	return (del != 0);     //Zwr false jeli ukad nie odpowiedzia na RESET
}

uint16_t SPI_VS_io_word(uint16_t word)
{
	uint16_t ret=SPI_RW_Byte(&VS_USART, word >> 8) << 8;  //Wylij bardziej znaczcy bajt
	ret+=SPI_RW_Byte(&VS_USART, word & 0xff);             //Wylij mniej znaczcy bajt
	return ret;
}

//Dostp do interfejsu polece VS1003
static inline void VS1003_xCS_Enable(_Bool en)
{
	if(en)
	{
		VS_PORT.OUTSET=VS_xDCS;    //Deaktywuj xDCS
		VS_PORT.OUTCLR=VS_xCS;     //Aktywuj xCS
	} else VS_PORT.OUTSET=VS_xCS;
}

//Dostp do interfejsu danych VS1003
static inline void VS1003_xDCS_Enable(_Bool en)
{
	if(en)
	{
		VS_PORT.OUTSET=VS_xCS;    //Deaktywuj xCS
		VS_PORT.OUTCLR=VS_xDCS;   //Aktywuj xDCS
	} else VS_PORT.OUTSET=VS_xDCS;
}

static inline void VS_1003_DREQ_High()
{
	while((VS_PORT.IN & VS_DREQ) == VS_DREQ);
}

static void VS1003_WriteReg(uint8_t cmd, uint16_t value)
{
	VS1003_xCS_Enable(true);       //Aktywuj DCI
	SPI_RW_Byte(&VS_USART, VS_WRITE_CMD);
	SPI_RW_Byte(&VS_USART, cmd);
	SPI_VS_io_word(value);         //Domylna konfiguracja
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();           //Zaczekaj na koniec realizacji polecenia
}

static uint16_t VS1003_ReadReg(uint8_t cmd)
{
	while(VS_USART.STATUS & USART_RXCIF_bm) VS_USART.DATA;   //Oprnij bufor odbiornika ze mieci
	VS1003_xCS_Enable(true);       //Aktywuj DCI
	SPI_RW_Byte(&VS_USART, VS_READ_CMD);
	SPI_RW_Byte(&VS_USART, cmd);
	uint16_t ret=SPI_VS_io_word(0);         //Domylna konfiguracja
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();           //Zaczekaj na koniec realizacji polecenia
	return ret;
}

_Bool VS1003_init()
{
	SPI_VS_init(2000);                         //Inicjalizacja interfejsu z niskim taktowaniem

	if(VS1003_Reset() == false) return false; //Zresetuj ukad

	VS1003_AddCommand(VS_WRITE_CMD, VS_MODE, (uint16_t*)&((VS_SCI_MODE){.SM_SDINEW=true}));  //Domylna konfiguracja
	VS1003_AddCommand(VS_WRITE_CMD, VS_BASS, &(uint16_t){0x0000});        //Nie uywamy "polepszaczy"
	VS1003_AddCommand(VS_WRITE_CMD, VS_CLOCKF, (uint16_t*)&((VS_SCI_CLOCKF){.SC_MULT=SC_MULT_XTAL30, .SC_ADD=SC_ADD_Mod15, .SC_FREQ=0}));   //XTALI=12.288MHz * 3
	VS1003_AddCommand(VS_WRITE_CMD, VS_VOL, &(uint16_t){VS_Volume | (VS_Volume << 8)});  //Ustaw nowe tumienie
	SPI_VS_init(F_CPU/7000);                 //Inicjalizacja interfejsu z maksymalnym dla VS taktowaniem
	VS_Buffer_ptr=0;   //Zaczynamy na pocztku bufora
	return true;
}

//Skonfiguruj DMA do transferu danych w odpowiedzi na aktywacj DREQ
void VS1003_DMA_init()
{
	DMA.CH3.CTRLA=0; //Zablokuj i zresetuj kana
	while(DMA.CH3.CTRLA & DMA_CH_ENABLE_bm);
	DMA.CH3.CTRLA=DMA_CH_RESET_bm;
	while(DMA.CH3.CTRLA & DMA_CH_RESET_bm);
	DMA.CH3.SRCADDR0=(uint16_t)&sound_file.buf & 0xff;         //Wykorzystujemy bufor na prbki przypisany plikowi
	DMA.CH3.SRCADDR1=((uint16_t)&sound_file.buf >> 8) & 0xff;
	DMA.CH3.SRCADDR2=0;
	DMA.CH3.DESTADDR0=(uint16_t)&USARTC1_DATA & 0xff;
	DMA.CH3.DESTADDR1=((uint16_t)&USARTC1_DATA >> 8) & 0xff;
	DMA.CH3.DESTADDR2=0;

	DMA.CH3.TRFCNT=(f_size(&sound_file) > _MAX_SS)? _MAX_SS : f_size(&sound_file);    //Ile bajtw danych zawiera bufor

	DMA.CH3.TRIGSRC=DMA_CH_TRIGSRC_EVSYS_CH2_gc;
	DMA.CH3.CTRLB=DMA_CH_TRNINTLVL_MED_gc;           //Przerwanie po zakoczeniu transferu danych
	DMA.CH3.ADDRCTRL=DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTRELOAD_NONE_gc | DMA_CH_DESTDIR_FIXED_gc;
	DMA.CTRL|=DMA_ENABLE_bm;
}

ISR(DMA_CH3_vect)
{
	_Bool OneCmd;

	_Bool VS_SendCmd()
	{
		SPI_Transact *trans;
		_Bool ret=false;
		if((trans=SPI_cbRead()) != NULL)       //Sprawd, czy mamy jakie polecenia
		{
			if(OneCmd) VS_1003_DREQ_High();               //Zaczekaj na gotowo ukadu
			OneCmd=true;
			VS1003_xCS_Enable(true);           //Odblokuj interfejs polece
			SPI_RW_Byte(&VS_USART, trans->data[0]);
			SPI_RW_Byte(&VS_USART, trans->data[1]);

			if(trans->data[0] == VS_WRITE_CMD)
			{
				SPI_RW_Byte(&VS_USART, trans->data[2]);  //Zapisz warto rejestru
				SPI_RW_Byte(&VS_USART, trans->data[3]);
			} else
			{
				while(VS_USART.STATUS & USART_RXCIF_bm) VS_USART.DATA;    //Oprnij bufor odbiornika
				trans->data[2]=SPI_RW_Byte(&VS_USART, 0x00);  //Odczytaj wybrany wczeniej rejestr
				trans->data[3]=SPI_RW_Byte(&VS_USART, 0x00);
			}
			SPI_FinishTrans(trans);            //Transakcja obsuona
			VS1003_xDCS_Enable(true);          //Odblokuj interfejs danych
			ret=true;
		}
		return ret;
	}

	UINT br;   //Liczba odczytanych bajtw
	char buf;  //Bufor na odczytany bajt

	DMA.CH3.CTRLB|=DMA_CH_TRNIF_bm;  //Skasuj flag przerwania

	OneCmd=false;
	while(VS_SendCmd());  //Mamy jakie polecenie do wysania?

	if(VS_Status > VS_NoPlaying)
	{
		VS_Buffer_ptr+=_MAX_SS;
		if(VS_Buffer_ptr >= f_size(&sound_file))   //Koniec pliku, koniec grania
		{
			f_close(&sound_file);
			VS_Status=VS_NoPlaying;
			VS1003_xDCS_Enable(false);    //Wycz ukad
			TCC1.CTRLA=TC_CLKSEL_OFF_gc;  //Wycz timer
			return;       //Koniec obsugi przerwania
		}
		f_lseek(&sound_file, VS_Buffer_ptr);
		f_read(&sound_file, &buf, 1, &br);  //Zapenij bufor danymi
		DMA.CH3.TRFCNT=((VS_Buffer_ptr - f_size(&sound_file)) > _MAX_SS)? _MAX_SS : (VS_Buffer_ptr - f_size(&sound_file));
		DMA.CH3.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc;  //Odblokuj kolejny transfer DMA
	}
}

void VS_DREQ_init()
{
	VS1003_xDCS_Enable(true);                   //Odblokuj interfejs danych ukadu
	DMA.CH3.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc;  //Odblokuj pierwszy transfer DMA

	EVSYS_CH7MUX=VS_DREQCH;                            //z pinu DREQ transmitowany do EVCH7

	TCC1.PER=(VS_USART.BAUDCTRLA+1) * 16 + 1;    //Co ile nadawa kolejne dane
	TCC1.CTRLB=TC_WGMODE_NORMAL_gc;
	TCC1.CTRLD=TC_EVACT_RESTART_gc | TC_EVSEL_CH7_gc;  //Kana 7 zeruje licznik
	TCC1.CTRLA=TC_CLKSEL_DIV1_gc;                      //Taktowanie timera
	EVSYS_CH2MUX=EVSYS_CHMUX_TCC1_OVF_gc;
}

void VS1003_StartPlay()
{
	UINT br;   //Liczba odczytanych bajtw
	char buf;

	if(VS1003_init())
	{
		VS_Status=VS_Playing;
		f_read(&sound_file, &buf, 1, &br);     //Zapenij bufor danymi
		VS1003_DMA_init();       //Zainicjuj kontroler DMA
		VS_DREQ_init();          //Zainicjuj przerwania od DREQ
	}
}

void VS1003_SineTest()
{
	VS1003_xCS_Enable(true);
	VS1003_WriteReg(VS_MODE, *(uint16_t*)&((VS_SCI_MODE){.SM_SDINEW=true, .SM_TESTS=true}));
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();

	VS1003_xDCS_Enable(true);
	SPI_VS_io_word(0x53ef);
	SPI_VS_io_word(0x6e7e);
	SPI_VS_io_word(0x0000);
	SPI_VS_io_word(0x0000);
	VS1003_xDCS_Enable(false);
}

void VS1003_AddCommand(uint8_t rw, uint8_t cmd, uint16_t *val)
{
	if(VS_Status > VS_NoPlaying)
	{
		SPI_Transact trans={.SelfDel=false, .Ready=false, .data[0]=rw, .data[1]=cmd, .data[2]=(*val >> 8) & 0xFF, .data[3]=*val & 0xFF};
		SPI_cbAdd(&trans);
		while(trans.Ready == false);
		if(rw == VS_READ_CMD) *val=trans.data[3] + (trans.data[2] << 8);
	} else
	{
		if(rw == VS_WRITE_CMD) VS1003_WriteReg(cmd, *val); else *val=VS1003_ReadReg(cmd);
	}
}