/*
 * SD.c
 *
 * Created: 2013-07-10 11:47:53
 *  Author: tmf
 */

#include "SD.h"
#include <avr/io.h>
#include <stdbool.h>
#include <string.h>
#include <util/delay.h>

SD_CardStatus SD_Status;               //Co aktualnie jest podpite

void SPI_SD_CS(_Bool SS_en)
{
	if(SS_en)
	{
		SD_PORT.OUTCLR=SD_CS;  //Aktywuj kart SD
		SD_WaitForReady();       //Zaczekaj a karta bdzie gotowa
	}
	else
	{
		SD_PORT.OUTSET=SD_CS;
		SPI_RW_Byte(0xff);        //Karta ulega deaktywacji po otrzymaniu kolejnego zbocza sygnau SCK
	}
}

void SPI_SetBaudrate_400kHz()
{
#ifdef SPI_USART
	uint16_t baud=F_CPU/400000UL;
	SD_USART.BAUDCTRLA=baud & 0xff;
	SD_USART.BAUDCTRLB=(baud >> 8) & 0xff;
#else
	SD_SPI.CTRL=(SD_SPI.CTRL & (~SPI_PRESCALER_gm)) | SPI_PRESCALER_DIV64_gc;  //Preskaler naley dobra zalenie od taktowania CPU
#endif
}

void SPI_SetMaxBaudrate()
{
#ifdef SPI_USART
	SD_USART.BAUDCTRLA=0;
	SD_USART.BAUDCTRLB=0;                     //Fclk=FPER/2 - maksymalne taktowanie SPI
#else
	SD_SPI.CTRL=(SD_SPI.CTRL & (~SPI_PRESCALER_gm)) | SPI_PRESCALER_DIV4_gc | SPI_CLK2X_bm;
#endif
}

uint8_t SPI_RW_Byte(uint8_t byte)
{
#ifdef SPI_USART
	SD_USART.DATA=byte;
	while(!(SD_USART.STATUS & USART_TXCIF_bm));
	SD_USART.STATUS=USART_TXCIF_bm;
	return SD_USART.DATA;
#else
	SD_SPI.DATA=byte;
	while(!(SD_SPI.STATUS & SPI_IF_bm));
	SD_SPI.STATUS=SPI_IF_bm;
	return SD_SPI.DATA;
#endif
}

void SPIInit()
{
	SD_PORT.DIRSET = SD_MOSI | SD_SCK;      //MOSI i SCK s wyjciami
#ifdef SPI_USART
	SPI_SetMaxBaudrate();
	SD_USART.CTRLC=USART_CMODE_MSPI_gc;       //Tryb SPI 0
	SD_USART.CTRLB=USART_TXEN_bm | USART_RXEN_bm;
#else
	SD_SPI.CTRL=SPI_ENABLE_bm | SPI_MASTER_bm | SPI_MODE_0_gc | SPI_PRESCALER_DIV64_gc;
#endif
}

_Bool SD_WaitForReady()
{
	uint16_t counter=5000;              //Czekamy maksymalnie ok. 500 ms
	SPI_RW_Byte(0xff);
	do									//Zaczekaj na gotowo karty - sygna CS musi by wczeniej aktywowany
	{
		if(SPI_RW_Byte(0xff) == 0xff) break;
		_delay_us(100);
	} while (--counter);
	return counter ? true : false;      //Czy wystpi timeout?
}

void SD_CRC7(uint8_t *crc, uint8_t byte)  //Wylicza CRC7 dla karty, pocztkowo crc=0
{
	for (uint8_t i=0; i < 8; i++)
	{
		*crc <<= 1;
		if ((((byte & 0x80) ^ (*crc & 0x80)) != 0))
		{
			*crc ^= 0x09;
		}
		byte <<= 1;
	}
}

uint8_t SD_SendCMD(uint8_t cmd, uint32_t arg)
{
	uint8_t crc=0, res;

	if(cmd & 0x80)
	{
		cmd&=0x7F;            //Skasuj najstarszy bit polecenia
		res=SD_SendCMD(CMD55, 0);  //Kolejne polecenie naley do grupy ACMD
		if(res > 0x01) return res; //Polecenie CMD55 zakoczyo si bdem
	}
	SPI_SD_CS(false);         //Deaktywuj kart
	SPI_SD_CS(true);          //Aktywuj kart

	cmd|=0x40;                //najstarsze dwa bity zawsze s rwne 01
	SPI_RW_Byte(cmd);
	SD_CRC7(&crc, cmd);
	for(uint8_t i=0; i<4; i++)
	{
		SPI_RW_Byte((arg >> 24) & 0xff);
		SD_CRC7(&crc, (arg >> 24) & 0xff);
		arg<<=8;
	}
	crc=(crc << 1) | 1;        //CRC7 dla SD jest przesunite o jeden bit w lewo i ma ustawiony najmodszy bit
	SPI_RW_Byte(crc);          //Wylij CRC polecenia

	uint8_t i=10;         //Odpowied moe nadej od 1 do 10 bajtw po poleceniu
	do
	{
		res=SPI_RW_Byte(0xff);
	} while ((res & 0x80) && --i);
	return res;                //Po wysaniu polecenia karta pozostaje wybrana w celu odczytu/zapisu kolejnych bajtw
}

uint32_t SD_GetR7()  //Pobierz odpowied typu R7
{
	uint32_t R7=0;
	for(uint8_t i=0; i<4; i++)
	{
		R7<<=8;
		R7|=SPI_RW_Byte(0xff);
	}
	return R7;
}

_Bool SD_GetResponse(void *buf, uint8_t size)
{
	uint8_t i=10;
	while((SPI_RW_Byte(0xff)!=0xfe) && (--i));
	if(!i) return false;
	while(size)
	{
		((uint8_t*)buf)[size-1]=SPI_RW_Byte(0xff);
		--size;
	}
	return true;
}

_Bool SD_CardInitV1()
{
	uint8_t i;
	for(i=0; i<255; i++)
	{
	  if(SD_SendCMD(ACMD41, 0) == 0) break;  //Karta w tryb aktywny
		else _delay_ms(10);
	}
	if(i == 255) return false;      //Karta nie akceptuje polecenia - uszkodzona?
	SD_GetR7();                     //Co prawda mamy odpowied R3, ale jej dugo jest taka sama jak R7
	SD_SendCMD(CMD16, 512);         //Dugo bloku 512 bajtw
	SD_Status=SD_SD;
	return true;
}

_Bool SD_CardInitV2()
{
	uint8_t i;

	if(SD_GetR7() != 0x1aa)
	 return false;  //Bedne napicie pracy karty

	for(i=0; i<255; i++)
	{
		if(SD_SendCMD(ACMD41, SD_OCR_CCS) == 0) break;  //Karta w tryb aktywny
		   else _delay_ms(40);

	}
	if(i == 255) return false;           //Karta nie akceptuje polecenia - uszkodzona?
	if(SD_SendCMD(CMD58, 0)==0)
	{
		if(SD_GetR7() & SD_OCR_CCS) SD_Status=SD_SDHC;  //Pobierz OCR
		    else SD_Status=SD_SD;
	}
	return true;
}

_Bool SD_CardInit()
{
	SPI_SetBaudrate_400kHz(&SD_USART);     //Kart naley zainicjowa przy SCK rwnym 400 kHz
	SPI_SD_CS(false);                      //Deaktywuj kart

	for(uint8_t i=0; i<10; i++)
	{
		SPI_RW_Byte(0xff);                 //Wylij co najmniej 74 takty SCK dla inicjalizacji karty
	}

	uint8_t i=10, ret;
	do
	{
		ret=SD_SendCMD(CMD0, 0);

	} while ((ret != SD_R1_Idle) && --i);  //Zaczekaj na prawidow odpowied ze strony karty

	if(!i)
	{
		SD_Status=SD_NoCard;               //Brak karty lub inny bd
		return false;                      //Niedoczekalimy si
	}

	i=10;
	do
	{
		ret=SD_SendCMD(CMD8, 0x1aa);
	} while ((ret & 0xc0) && --i);

	_Bool result;
	if(ret==0x05) result=SD_CardInitV1(); else result=SD_CardInitV2(); //Z jak kart mamy do czynienia
	SPI_SetMaxBaudrate(&USARTC1);  //Przecz interfejs na maksymaln szybko pracy
	return result;
}

_Bool SD_disk_read (uint8_t *buff, uint32_t sector, uint8_t count)
{
	if(SD_Status == SD_NoCard) return false;

	if (SD_Status == SD_SD) sector*=512;	//SDSC adresowana jest bajtami

	if(count == 1)
	{	//Odczytujemy pojedynczy sektor
		if((SD_SendCMD(CMD17, sector) == 0) && SD_Rec_Datablock(buff, 512))
		count=0;
	} else
	{				//Odczytujemy kilka sektorw na raz
		if(SD_SendCMD(CMD18, sector) == 0)
		{
			do
			{
				if(!SD_Rec_Datablock(buff, 512)) break;
				buff+= 512;
			} while(--count);
			SD_SendCMD(CMD12, 0);				//Koniec transmisji
		}
	}

	SPI_SD_CS(false);
	return count ? false : true;
}

_Bool SD_Rec_Datablock(uint8_t *buff, uint16_t size)
{
	uint8_t rec, timeout;

	timeout=200;
	do
	{
		_delay_us(500);                  //Zanim blok danych bdzie gotowy potrzebujemy chwilk
		rec=SPI_RW_Byte(0xff);
	} while((rec == 0xFF) && timeout--); //Czekamy na otrzymanie 0xfe, ale max 100 ms

	if(rec != 0xfe) return false;		 //Bd - nie ma pocztku danych

	do {						         //Odbierz blok danych
		*buff=SPI_RW_Byte(0xff);
		buff++;
	} while(--size);
	SPI_RW_Byte(0xff);                   //Odbierz CRC bloku danych - obowizkowe
	SPI_RW_Byte(0xff);
	return true;
}