/*
 * Composite_overlay_LM1881.c
 *
 * Created: 2012-05-01 12:40:49
 *  Author: tmf
 */ 

#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdbool.h>
#include <avr/sleep.h>

#include "Font8x8.h"                             //Tablica czcionek 8x8

#define VIDEO_Scan_line       64                 //Czas trwania caej linii
#define VIDEO_Sync_Len       4.7                 //Czas synchronizacji
#define VIDEO_Back_porch     5.7                 //Czas od koca synchronizacji do pocztku wywietlania pikseli
#define VIDEO_Front_porch   1.65                 //Czas od koca wywietlania pikseli do impulsu synchronizacji

#define VIDEO_ACTIVE_PIXEL_START   2             //Pocztek widocznego obszaru linii
#define VIDEO_ACTIVELINE_START    60             //Pierwsza wywietlana linia
#define VIDEO_ACTIVELINE_END     260             //Ostatnia wywietlana linia

#define VIDEO_TXTCOLUMNS     40                  //Dugo linii tekstu (w znakach)
#define VIDEO_TXTLINES       ((VIDEO_ACTIVELINE_END - VIDEO_ACTIVELINE_START)/8)                  //Liczba linii tekstu, liczba linii wywietlanych przez wysoko znaku

#define US_2_CNT( TIME_US ) (((F_CPU / 1000000L) * (TIME_US) ) - 1)               //Przelicz mikrosekundy na tyknicia timera

uint8_t video_linebuf[VIDEO_TXTCOLUMNS];           //Bufor linii video
uint8_t framebuf[VIDEO_TXTLINES*VIDEO_TXTCOLUMNS]; //Bufor ekranu

bool OSC_wait_for_rdy(uint8_t clk)
{
	uint8_t czas=255;
	while ((!(OSC.STATUS & clk)) && (--czas)) // Czekaj na ustabilizowanie si generatora
	 _delay_ms(1);
	return czas;   //false jeli generator nie wystartowa, true jeli jest ok
}

bool RC32M_en()
{
	OSC.CTRL |= OSC_RC32MEN_bm; //Wcz generator RC 32 MHz
	return OSC_wait_for_rdy(OSC_RC32MEN_bm); //Zaczekaj na jego poprawny start
}

bool EXCLK_en()
{
	OSC_CTRL |= OSC_XOSCEN_bm;
	return OSC_wait_for_rdy(OSC_XOSCEN_bm);
}

void SelectPLL(OSC_PLLSRC_t src, uint8_t mult)
{
	mult&=OSC_PLLFAC_gm;
	OSC.PLLCTRL=src | mult;              //Ustaw rdo i mno?nik PLL
	OSC.CTRL|=OSC_PLLEN_bm;				 //Wcz uk?ad PLL
	OSC_wait_for_rdy(OSC_PLLRDY_bm);     //Poczekaj na ustabilizowanie si PLL
}


uint16_t HLine;               //Licznik linii
uint16_t addr;                //Warto addr musi by przechowywana pomidzy kolejnymi wywoaniami przerwania

ISR(PORTA_INT0_vect)  //Przerwanie V-Sync
{
  HLine=0;  //Pocztek nowej linii
  addr=-VIDEO_TXTCOLUMNS; //W linii numer 0 zostanie dodane VIDEO_TXTCOLUMNS, w efekcie addr bdzie mia warto 0	
}

ISR(PORTA_INT1_vect)  //Przerwanie H-Sync (composite)
{
	TCC0_CNT=0;
}

ISR(TCC0_CCA_vect)
{
	HLine++;
	if((HLine>=VIDEO_ACTIVELINE_START) && (HLine<VIDEO_ACTIVELINE_END))
	{
		USARTC0_DATA=255;                 //Wypenij wstpnie bufor - unikamy jitteru zwizanego ze startem DMA
		DMA_CH0_CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc;  //Odblokuj transfer kolejnej linii danych

		uint8_t *bufaddr=video_linebuf;    
		uint8_t chbyte, znak;
		
		uint8_t chline=(HLine-VIDEO_ACTIVELINE_START) & 0b00000111;    //Numer linii znaku
		if(chline==0) addr+=VIDEO_TXTCOLUMNS;                   //Ustaw wskanik na now lini w buforze ramki
		
		uint8_t *rowstart=(uint8_t*) &font8x8[chline*95];       //95 - liczba znakw w tablicy znakw
        uint16_t tmpaddr=addr;
		
		for(uint8_t indeks=0; indeks<VIDEO_TXTCOLUMNS; indeks++)
	    {    //Przepisz jedn lini znakw do bufora linii, ktry aktualnie nie jest wywietlany
		 znak=framebuf[tmpaddr++];
		 chbyte=pgm_read_byte(&rowstart[znak]);
		 *(bufaddr++)=chbyte;             //Przepisz map znaku do bufora nowej linii
        }
	}
}

void DotClockTimer_init()      //Timer okrelajcy czas trwania pikseli
{
	TCC0.CCA=((VIDEO_Sync_Len + VIDEO_Back_porch + VIDEO_ACTIVE_PIXEL_START)*F_CPU)/1000000UL; //Czas do rozpoczcia wywietlania pikseli - aktywacji DMA
	TCC0.CTRLB=TC0_CCAEN_bm | TC_WGMODE_SS_gc;     //Single slope mode
	TCC0.INTCTRLB=TC_CCAINTLVL_HI_gc;              //Odblokuj przerwania pocztku aktywnego obszaru linii
	TCC0.CTRLA=TC_CLKSEL_DIV1_gc;                  //Timer taktowany z CLKPER
}

void DMA_init()
{
	DMA.CTRL=DMA_ENABLE_bm | DMA_PRIMODE_CH0RR123_gc;  //Odblokuj kontroler DMA, kana 0 ma najwyszy priorytet, reszta round robin
	
	DMA.CH0.ADDRCTRL=DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTDIR_FIXED_gc | DMA_CH_DESTRELOAD_BURST_gc; //Zwikszamy adres rda i przeznaczenia, reload adresu co blok
	DMA.CH0.TRFCNT=VIDEO_TXTCOLUMNS;                       //Blok ma dugo jednej linii obrazu
	
	DMA_CH0_SRCADDR0=(uint16_t)video_linebuf & 0xFF;   //Adres pocztku transferu
	DMA_CH0_SRCADDR1=(uint16_t)video_linebuf >> 8;
	DMA_CH0_SRCADDR2=0;
	
	DMA.CH0.DESTADDR0=(uint16_t)&USARTC0.DATA & 0xFF;      //Dane bd wysyane do USART
	DMA.CH0.DESTADDR1=(uint16_t)&USARTC0.DATA >> 8;
	DMA.CH0.DESTADDR2=0;    
	DMA.CH0.TRIGSRC=DMA_CH_TRIGSRC_USARTC0_DRE_gc;         //Pusty rejestr nadajnika inicjuje transfer DMA
	DMA.CH0.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc;  //Kana 0 dugo transferu 1 bajt
}

void USART_SPI_init()
{
	PORTC.PIN3CTRL=PORT_INVEN_bm | PORT_OPC_WIREDOR_gc; //Musimy znaegowa sygna, aby port spoczynkowo mia stan 0, stan 0 jest pasywny (HiZ)
	PORTC.DIRSET=(1<<3);                            //MOSI (Txd) jest wyjciem
	USARTC0.BAUDCTRLA=1;                            //Fout=8 MHz
	USARTC0.CTRLB=USART_TXEN_bm;
	USARTC0.CTRLC=USART_CMODE_MSPI_gc;              //Tryb SPI, LSB first		
}

void IO_init()     ///Zainicjuj obsug przerwa H i V-sync
{
	PORTA_INTCTRL=PORT_INT1LVL_HI_gc | PORT_INT0LVL_HI_gc;  //Wysoki priorytet przerwania z portu A
	PORTA.INT0MASK=(1<<4);      //V-Sync
	
	PORTA.INT1MASK=(1<<3);      //H-Sync
	PORTCFG_MPCMASK=(1<<4) | (1<<3);  //Zmieniamy na raz stan dwch pnw IO
	PORTA.PIN4CTRL=PORT_ISC_FALLING_gc;  //Przerwanie generujemy na zboczu opadajcym
}

void INT_init()
{
	PMIC_CTRL|=PMIC_HILVLEN_bm;     //Odblokuj przerwania niskiego poziomu
	sei();
}

int main(void)
{
	RC32M_en();                       //Odblokuj wewntrzny generator RC
	
	EXCLK_en();                         //Odblokuj zegar zewntrzny doprowadzony do XTAL1
    SelectPLL(OSC_PLLSRC_XOSC_gc, 8);   //FXTAL1=4 MHz * 8 = 32 MHz


	CPU_CCP=CCP_IOREG_gc;            //Odblokuj zmian konfiguracji
	//CLK.CTRL=CLK_SCLKSEL_RC32M_gc;   //Wybierz generator RC 32MHz
	CLK.CTRL=CLK_SCLKSEL_PLL_gc;     //Wybierz PLL
	
	for(uint8_t y=0; y<VIDEO_TXTLINES; y++)
	  for(uint8_t x=0; x<VIDEO_TXTCOLUMNS; x++) framebuf[y*VIDEO_TXTCOLUMNS+x]=(x+y) % 96;
	
	DotClockTimer_init();            //Zainicjuj timer zliczjcy piksele
	USART_SPI_init();
	IO_init();
	DMA_init();
	INT_init();
	
	set_sleep_mode(SLEEP_MODE_IDLE);   //Procesor bdzie wchodzi w stan Idle
	
	while(1)
    {
		sleep_mode();                  //Upij procesor
    }
}