/*
 * ADPCM_IMA.c
 *
 * Created: 2013-10-08 22:04:47
 *  Author: tmf
 */


#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdbool.h>

#define GET_FAR_ADDRESS(var)                  \
({                                            \
	__uint24 tmp;                             \
	__asm__ __volatile__(                     \
	"ldi	%A0, lo8(%1)"           "\n\t"    \
	"ldi	%B0, hi8(%1)"           "\n\t"    \
	"ldi	%C0, hh8(%1)"           "\n\t"    \
	: "=d" (tmp) :	"p"  (&(var)));           \
	tmp;                                      \
})

extern __memx const uint8_t _binary____intro_adpcm_ima_stripped_end;
extern const __uint24 _binary____intro_adpcm_ima_stripped_size;
extern __memx const uint16_t _binary____intro_adpcm_ima_stripped_start;

#define Snd_NoPlaying      0        //Nic nie jest odgrywane
#define Snd_Playing8Bit    1        //Odtwarzamy 8-bitowe prbki

const uint8_t __memx *srcaddr;      //Adres prbek dwikowych
__uint24 samplesize;                //Dugo odtwarzanej prbki w bajtach
volatile uint8_t isPlaying;         //Stan odtwarzacza

//* Tabela zmian indeksw
const int8_t __flash IndexTable[16] = {
	-1, -1, -1, -1, 2, 4, 6, 8,
	-1, -1, -1, -1, 2, 4, 6, 8
};

// Tablica wartoci kwantyzatora
const int __flash StepSizeTable[89] = {
	7, 8, 9, 10, 11, 12, 13, 14, 16, 17,
	19, 21, 23, 25, 28, 31, 34, 37, 41, 45,
	50, 55, 60, 66, 73, 80, 88, 97, 107, 118,
	130, 143, 157, 173, 190, 209, 230, 253, 279, 307,
	337, 371, 408, 449, 494, 544, 598, 658, 724, 796,
	876, 963, 1060, 1166, 1282, 1411, 1552, 1707, 1878, 2066,
	2272, 2499, 2749, 3024, 3327, 3660, 4026, 4428, 4871, 5358,
	5894, 6484, 7132, 7845, 8630, 9493, 10442, 11487, 12635, 13899,
	15289, 16818, 18500, 20350, 22385, 24623, 27086, 29794, 32767
};


int8_t ADPCM_index;              //Indeks do tablicy wielkoci krokw
__int24 ADPCM_predsample;        //Predyktor

// ADPCMDecoder - funkcja dekodera ADPCM
// Wyjcie to 16-bitowa prbka w kodzie U2

int16_t ADPCMDecoder(uint8_t code)
{
	int ADPCM_step;                  //Krok kwantyzatora
	int ADPCM_diffq;                 //Przewidywana warto

	ADPCM_step = StepSizeTable[ADPCM_index];
	ADPCM_diffq=ADPCM_step >> 3;
	if(code & 4) ADPCM_diffq+=ADPCM_step;
	if(code & 2) ADPCM_diffq+=ADPCM_step >> 1;
	if(code & 1) ADPCM_diffq+=ADPCM_step >> 2;
	if(code & 8) ADPCM_predsample-=ADPCM_diffq; else ADPCM_predsample+=ADPCM_diffq;
	if(ADPCM_predsample > 32767) ADPCM_predsample=32767;            //Musimy zadba o odpowiednie "zawijanie" zmiennej
		else if(ADPCM_predsample < -32768) ADPCM_predsample=-32768;
	ADPCM_index += IndexTable[code];   //Nowy krok kwantyzatora
	if(ADPCM_index < 0) ADPCM_index=0; //Zawi indeks
	if(ADPCM_index > 88) ADPCM_index=88;
	return ADPCM_predsample;
}

bool OSC_wait_for_rdy(uint8_t clk)
{
	uint8_t czas=255;
	while ((!(OSC.STATUS & clk)) && (--czas)) // Czekaj na ustabilizowanie si generatora
	_delay_ms(1);
	return czas;   //false jeli generator nie wystartowa, true jeli jest ok
}

bool OSC_init()
{
	OSC.CTRL |= OSC_RC32MEN_bm;          //Wcz generator RC 32 MHz
	if(OSC_wait_for_rdy(OSC_RC32MEN_bm)) //Zaczekaj na jego poprawny start
	{
		OSC.PLLCTRL=OSC_PLLSRC_RC32M_gc | 16; //PLL taktowany 8 MHz i mnonik 16-razy, na wyjciu 128 MHz
		OSC.CTRL|=OSC_PLLEN_bm;               //Wcz PLL
		if(OSC_wait_for_rdy((OSC_PLLEN_bm)))  //Poczekaj na jego stabilizacj
		{
			CPU_CCP=CCP_IOREG_gc;            //Konfiguruj preskalery
			CLK_PSCTRL=CLK_PSADIV_1_gc | CLK_PSBCDIV_4_1_gc;  //CLKPER4=128 MHz, CLKPER=32 MHz

			asm volatile ("nop");          //Niezbdne - patr tekst
			asm volatile ("nop");
			CPU_CCP=CCP_IOREG_gc;
			CLK_CTRL=CLK_SCLKSEL_PLL_gc;   //Wybierz zegar generowany przez PLL (128 MHz)
			return true;
		}
	}
	return false;
}

ISR(TCC1_OVF_vect)
{
	static uint8_t nibble=1;

	uint8_t smp=*srcaddr;
	if(nibble == 1) smp>>=4; else srcaddr++;
	smp&=0x0f;   //Kod ADPCM to tylko 4 bity
	nibble^=1;
	TCF0_CCABUF=(ADPCMDecoder(smp) + 0x8000) >> 2; //PWM ma tylko 14 bitow rozdzielczo
	samplesize--;
	if(samplesize == 0)
	{
		isPlaying=Snd_NoPlaying;
		TCF0_CTRLA=TC_CLKSEL_OFF_gc;  //Koniec odtwarzania dwikw
		TCC1_CTRLA=TC_CLKSEL_OFF_gc;
	}
}

void Snd_init(const uint8_t __memx *smpaddr, __uint24 smpsize, uint16_t bitrate, _Bool HiRes)
{
	void Timer_init(uint16_t samplerate)  //Generuje zdarzenia wywoujce konwersj DAC i przeczanie buforw
	{
		PORTF_DIRSET=PIN0_bm;      //Wyjcie CCA

		HIRESF_CTRLA=HIRES_HREN_TC0_gc;  //Odblokuj HiRes dla TCF0
		//W nowszych XMEGA mamy jeszcze bit HIRES_HRPLUS_bm - HiRes x8

		TCF0.PER=0x4000;   //14 bitw rozdzielczoci PWM
		TCF0.CTRLB=TC_WGMODE_SS_gc | TC0_CCAEN_bm;
		TCF0.CTRLA=TC_CLKSEL_DIV1_gc;        //128 MHz

		TCC1.PER=F_CPU/samplerate-1;
		TCC1.CTRLB=TC_WGMODE_NORMAL_gc;
		TCC1.INTCTRLA=TC_OVFINTLVL_LO_gc;
		TCC1.CTRLA=TC_CLKSEL_DIV1_gc;
	}

	samplesize=smpsize*2;         //Dugo odtwarzanej prbki w bajtach
	srcaddr=smpaddr;              //Adres prbki

	ADPCM_predsample=*srcaddr++;       //Mniej znaczcy bajt prbki
	ADPCM_predsample=*srcaddr++ << 8;  //Bardziej znaczcy bajt prbki
	ADPCM_index=*srcaddr++;            //Pocztkowy indeks do tablicy krokw

	srcaddr++;
	samplesize-=4;

	Timer_init(bitrate);          //Zainicjuj timer z odpowiednim samplerate
	isPlaying=Snd_Playing8Bit;
}

int main(void)
{
	OSC_init();                 //Taktowanie CLKPER4=128 MHz, CLKPER=32 MHz
	isPlaying=Snd_NoPlaying;
	PMIC_CTRL=PMIC_LOLVLEN_bm;  //Odblokuj przerwania najniszego poziomu
	sei();

	PORTD_PIN0CTRL=PORT_OPC_PULLUP_gc; //Wczamy pullup na SW0

	while(1)
	{
		if(((PORTD_IN & PIN0_bm) == 0) && (isPlaying == Snd_NoPlaying))
		Snd_init((const uint8_t __memx *)GET_FAR_ADDRESS(_binary____intro_adpcm_ima_stripped_start), GET_FAR_ADDRESS(_binary____intro_adpcm_ima_stripped_size), 8000, false);  //Zagraj prbk 8kHz/8 bitw
	}
}