/**
 * (c)2013 Tomasz Francuz
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 *    Atmel microcontroller product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <avr/interrupt.h>
#include <asf.h>
#include "conf_usb.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <string.h>
#include "SD.h"

//Makro umieszczajce zadany acuch w przestrzeni adresowej __flash
#define PGM_STR(X) ((const __flash char[]) { X })

typedef void (*CmdHandler)(char *param);

typedef struct
{
	const char __flash *cmd;                //Wskanik do polecenia w formacie ASCIIZ
	const CmdHandler  Handler;              //Funkcja realizujca polecenie
} Command;

void Cmd_InitSD(char *param);    //Inicjalizuje kart SD, naley wykona przed dostpem do niej
void Cmd_CID(char *param);       //Wywietl rejestr CID
void Cmd_CSD(char *param);       //Wylwietl rejestr CSD
void Cmd_Read(char *param);      //Wylwietl wskazany sektor - read nr_sektora, liczba sektorw

//Lista rozpoznawanych polece
const __flash Command __flash Polecenia[]={{PGM_STR("Init"), Cmd_InitSD}, {PGM_STR("CID"), Cmd_CID}, {PGM_STR("CSD"), Cmd_CSD},
	                                       {PGM_STR("read"), Cmd_Read}};

int main(void)
{
	void InterpretCommand(char *cmdline)
	{
		uint8_t indeks;
		char *cmd=strtok(cmdline, " \r\n"); //Wydziel polecenie z przekazanej linii
		uint8_t max_indeks=sizeof(Polecenia)/sizeof(Polecenia[0]);
		for(indeks=0; indeks<max_indeks; indeks++)
		{
			if(strcmp_P(cmd, Polecenia[indeks].cmd)==0) //Przeszukaj list polece
			{
				Polecenia[indeks].Handler(cmdline);   //Wywoaj funkcj obsugi przesanego polecenia
				break;
			}
		}
		if(indeks == max_indeks)  //Jeli polecenie nieznane...
		{
			udi_cdc_write_buf(cmd, strlen(cmd));
			udi_cdc_write_buf(" - nieznane polecenie\r\n", 22); //Bd
		}
	}

	SD_PORT.OUTSET=0xff;         //Ustaw ew. sygnay SS urzdze podczonych do SPI
	SD_PORT.DIRSET=SD_CS | F_CS | TP_CS; //Deaktywuj sygnay CS karty SD, kontrolera TP i pamici FLASH (na module LCD)
	SPIInit();

	PMIC.CTRL = PMIC_LOLVLEN_bm | PMIC_MEDLVLEN_bm | PMIC_HILVLEN_bm; //Wczamy wszystkie poziomy przerwa (potrzebujemy tylko najniszy)
	sei();

	sysclk_init();
	udc_start();

	while (1)
	{
		char Bufor[256];   //Bufor na odebranie linii
		uint8_t indeks=0;

		while(1)
		 {
			if(udi_cdc_is_rx_ready())
			{
				Bufor[indeks] = udi_cdc_getc();
				if(Bufor[indeks] == '\n')
				{
					if((indeks >= 1) && (Bufor[indeks-1] == '\r')) --indeks;
					Bufor[indeks]=0; //Wpisz kocowy znak NUL
					break;           //Odebrano pen lini
				}
				++indeks;
			}
		 }
		InterpretCommand(Bufor);  //Zdekoduj przesane polecenie
	}
}

void main_suspend_action(void)
{
//Interfejs wchodzi w tryb powerdown
}

void main_resume_action(void)
{
//Wybudzenie interfejsu
}

void main_sof_action(void)
{
//	Odebrano ramk
}

bool main_cdc_enable(uint8_t port)
{
	// Otwarto port CDC
	return true;
}

void main_cdc_disable(uint8_t port)
{
	// Zamknito port CDC
}

void main_cdc_set_dtr(uint8_t port, bool b_enable)
{
	if (b_enable) {
		// Host otwar port COM
	}else{
		// Host zamkn port COM
	}
}

void uart_rx_notify(uint8_t port)
{
	//Opcjonalnie powiadomienie o nadejciu znaku
}

void uart_config(uint8_t port, usb_cdc_line_coding_t * cfg)
{
	//Konfiguracja w odpowiedzi na otwarcie CDC
}

void udi_cdc_write_txt(char *txt)
{
	udi_cdc_write_buf(txt, strlen(txt));
}

void Cmd_InitSD(char *param)
{
	udi_cdc_write_txt("Inicjalizuje karte...\r\n");
	if(SD_CardInit()==false) udi_cdc_write_txt("Nie udalo sie zainicjalizowac karty...\r\n");
	 else
	 {
		 switch(SD_Status)
		 {
			 case SD_NoCard: break;    //Ju wczeniej zostao obsuone
			 case SD_SD :   udi_cdc_write_txt("Zainicjowano karte SD.\r\n"); break;
			 case SD_SDHC:  udi_cdc_write_txt("Zainicjowano karte SDHC.\r\n"); break;
		 }
	 }
}

void Cmd_CID(char *param)
{
	if(SD_Status==SD_NoCard)
	{
		udi_cdc_write_txt("Karta nie zostaa zainicjowana.\r\n");
		return;
	}

	char buf[16];
	SD_CIDReg *cid=(SD_CIDReg*)buf;
	uint8_t r1=SD_SendCMD(CMD10, 0);  //Odczytaj CID
	if(r1==0)
	{
		SD_GetResponse(cid->Reg, 16);
		char bufor[255];
		sprintf(bufor, "CID:\r\nMID:%d\r\nOID:%c%c\r\nPNM:%c%c%c%c%c\r\nSerial:%lu\r\nData produkcji (MM/YYYY): %d/%d\r\n", cid->MID, cid->OID[1], cid->OID[0],
		        cid->PNM[4], cid->PNM[3], cid->PNM[2], cid->PNM[1], cid->PNM[0], cid->PSN, cid->MDT_Month, 2000 + cid->MDT_Year1*10 + cid->MDT_Year2);
		udi_cdc_write_txt(bufor);
	}
}

void Cmd_CSD(char *param)
{
	if(SD_Status==SD_NoCard)
	{
		udi_cdc_write_txt("Karta nie zostaa zainicjowana.\r\n");
		return;
	}

	char buf[16];
	SD_CSDRegv1 *csd=(SD_CSDRegv1*)buf;
	uint8_t r1=SD_SendCMD(CMD9, 0);  //Odczytaj CSD
	if(r1==0)
	{
		SD_GetResponse(csd->Reg, 16);
		char bufor[128];
		sprintf(bufor, "CSD:\r\nDlugosc bloku:%lu\r\n", (1UL << csd->Read_Bl_Len));
		udi_cdc_write_txt(bufor);
		uint64_t size;  //Rozmiar karty
		if(csd->CSD_Struct == 0)
		{
			size=(csd->C_Size + 1) * (1ULL << csd->Read_Bl_Len) * (1UL << (csd->C_Size_mult + 2));
		} else
		{
			size=(((SD_CSDRegv2*)csd)->C_Size + 1) * 524288ULL;  //C_Size zawiera liczb blokw po 512 kB
		}
		sprintf(bufor, "Rozmiar karty:%lu kB.\r\n", (uint32_t)(size/1024));
		udi_cdc_write_txt(bufor);
	}
}

void Cmd_Read(char *param)
{
	param=strtok(NULL, " ,;");
	if(param==NULL) return;      //Bd
	uint32_t sector=atol(param);  //Pobierz nr pierwszego sektora
	param=strtok(NULL, " ,;");
	if(param==NULL) return;      //Bd
	uint16_t size=atoi(param);   //Ile sektorw odczyta?
	uint8_t *bufor=malloc(size * 512); //Przydziel pami na bufor
	if(bufor == NULL)
	{
		udi_cdc_write_txt("Brak miejsca na bufor.\r\n");
		return;
	}
	if(SD_disk_read(bufor, sector, size)) //Odczytaj dane
	{
		char str[3*16+3], chBuf[4];
		uint16_t ofs=0;
		uint8_t asize;
		size*=512;

		while(size)
		{
			str[0]=0;
			if(size>16) asize=16; else asize=size;
			for(uint8_t i=0; i<asize; i++)   //Zamie odczytane dane na wiersz znakw ASCII
			{
				sprintf(chBuf, "%02X ", bufor[ofs + i]);
				strcat(str, chBuf);
			}
			strcat(str, "\r\n");         //Dodaj znak koca wiersza
			udi_cdc_write_txt(str);      //Wylij dane do komputera
			size-=asize;
			ofs+=asize;
		}
	}
	free(bufor);
}